import { AD_STITCHING_GEOS, PREROLL_TIMEOUT } from 'ads/surestream/consts';
import { stringify } from 'query-string';
import * as PlayerType from 'util/player-type';
import { unitTest } from 'tests/utils/module';
import { CONTENT_MODE_VOD, VODTwitchContentStream } from 'stream/twitch-vod';
import { usherHost as USHER_HOST } from 'settings';

unitTest('stream | vod', function(hooks) {
    hooks.beforeEach(function() {
        this.vodId = `v${parseInt(QUnit.config.current.testId, 16)}`;
        this.oauthToken = `oauth_${QUnit.config.current.testId}`;
        this.usherParams = {
            godlike: true,
        };
        this.accessTokenParams = {
            // eslint-disable-next-line camelcase
            need_https: false,
            adblock: true,
            platform: 'web-test',
            // eslint-disable-next-line camelcase
            player_type: PlayerType.PLAYER_SITE,
        };
        this.experimentSettings = {
            realtimeQos: Promise.resolve('control'),
            fastBread: Promise.resolve('control'),
        };
        this.api.setLoggedIn(true);
    });

    QUnit.test('contentType should be vod', function(assert) {
        const stream = new VODTwitchContentStream(
            this.vodId,
            Promise.resolve(this.oauthToken),
            this.usherParams
        );

        assert.equal(stream.contentType, CONTENT_MODE_VOD);
    });

    QUnit.test('videoId should be the given value', function(assert) {
        const stream = new VODTwitchContentStream(
            this.vodId,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.nauthParams
        );

        assert.equal(stream.videoId, this.vodId);
    });

    QUnit.test('channel should return empty string', function(assert) {
        const stream = new VODTwitchContentStream(
            this.vodId,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.nauthParams
        );

        assert.equal(stream.channel, '');
    });

    QUnit.test('rtqos should be the given value', function(assert) {
        this.userInfo = this.api.expectUserInfo({
            geo: 'US',
        });
        const accessToken = this.api.expectVodAccessToken(this.vodId, {
            canAccess: true,
            accessTokenParams: this.accessTokenParams,
            oauthToken: this.oauthToken,
        });
        const oAuthResponse = Promise.resolve(this.oauthToken);

        this.experimentSettings = {
            realtimeQos: Promise.resolve('realtime'),
        };

        const stream = new VODTwitchContentStream(
            this.vodId,
            oAuthResponse,
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );

        const params = {
            nauth: accessToken.token,
            nauthsig: accessToken.sig,
            // eslint-disable-next-line camelcase
            allow_source: true,
            godlike: true,
            rtqos: 'realtime',
            /* eslint-disable camelcase */
            fast_bread: false,
            baking_bread: true,
            baking_brownies: true,
            baking_brownies_timeout: PREROLL_TIMEOUT,
            /* eslint-enable camelcase */
        };

        const streamUrl = stream.streamUrl.then(url => {
            assert.equal(
                url,
                // eslint-disable-next-line max-len
                `${USHER_HOST}/vod/${this.vodId.substring(1)}.m3u8?${stringify(params)}`
            );
        });
        return streamUrl;
    });

    QUnit.test('streamUrl should be the correct computed value', function(assert) {
        this.userInfo = this.api.expectUserInfo({
            geo: 'US',
        });
        const accessToken = this.api.expectVodAccessToken(this.vodId, {
            canAccess: true,
            accessTokenParams: this.accessTokenParams,
            oauthToken: this.oauthToken,
        });
        const oAuthResponse = Promise.resolve(this.oauthToken);

        const stream = new VODTwitchContentStream(
            this.vodId,
            oAuthResponse,
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );

        const params = {
            nauth: accessToken.token,
            nauthsig: accessToken.sig,
            // eslint-disable-next-line camelcase
            allow_source: true,
            godlike: true,
            rtqos: 'control',
            /* eslint-disable camelcase */
            fast_bread: false,
            baking_bread: true,
            baking_brownies: true,
            baking_brownies_timeout: PREROLL_TIMEOUT,
            /* eslint-enable camelcase */
        };

        const streamUrl = stream.streamUrl.then(url => {
            assert.equal(
                url,
                // eslint-disable-next-line max-len
                `${USHER_HOST}/vod/${this.vodId.substring(1)}.m3u8?${stringify(params)}`
            );
        });

        return streamUrl;
    });

    AD_STITCHING_GEOS.forEach(geo => {
        // eslint-disable-next-line max-len
        QUnit.test(`if in ${geo} then streamUrl should be the correct computed value with true baking_bread settings`, function(assert) {
            this.userInfo = this.api.expectUserInfo({
                geo: geo,
            });
            const accessToken = this.api.expectVodAccessToken(this.vodId, {
                canAccess: true,
                accessTokenParams: this.accessTokenParams,
                oauthToken: this.oauthToken,
            });
            const oAuthResponse = Promise.resolve(this.oauthToken);

            this.experimentSettings = {
                realtimeQos: Promise.resolve('control'),
            };

            const stream = new VODTwitchContentStream(
                this.vodId,
                oAuthResponse,
                this.usherParams,
                this.accessTokenParams,
                this.experimentSettings
            );

            const params = {
                nauth: accessToken.token,
                nauthsig: accessToken.sig,
                // eslint-disable-next-line camelcase
                allow_source: true,
                godlike: true,
                rtqos: 'control',
                /* eslint-disable camelcase */
                fast_bread: false,
                baking_bread: true,
                baking_brownies: true,
                baking_brownies_timeout: PREROLL_TIMEOUT,
                /* eslint-enable camelcase */
            };

            const streamUrl = stream.streamUrl.then(url => {
                assert.equal(
                    url,
                    // eslint-disable-next-line max-len
                    `${USHER_HOST}/vod/${this.vodId.substring(1)}.m3u8?${stringify(params)}`
                );
            });
            return streamUrl;
        });
    });

    QUnit.test('streamUrl should be the correct computed value if not in AD_STITCHING_GEOS', function(assert) {
        this.userInfo = this.api.expectUserInfo({
            geo: 'XX',
        });

        const accessToken = this.api.expectVodAccessToken(this.vodId, {
            canAccess: true,
            accessTokenParams: this.accessTokenParams,
            oauthToken: this.oauthToken,
        });
        const oAuthResponse = Promise.resolve(this.oauthToken);

        this.experimentSettings = {
            realtimeQos: Promise.resolve('control'),
        };
        const stream = new VODTwitchContentStream(
            this.vodId,
            oAuthResponse,
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );

        const params = {
            nauth: accessToken.token,
            nauthsig: accessToken.sig,
            // eslint-disable-next-line camelcase
            allow_source: true,
            godlike: true,
            rtqos: 'control',
            /* eslint-disable camelcase */
            fast_bread: false,
            baking_bread: false,
            baking_brownies: false,
            baking_brownies_timeout: PREROLL_TIMEOUT,
            /* eslint-enable camelcase */
        };

        const streamUrl = stream.streamUrl.then(url => {
            assert.equal(
                url,
                // eslint-disable-next-line max-len
                `${USHER_HOST}/vod/${this.vodId.substring(1)}.m3u8?${stringify(params)}`
            );
        });

        return streamUrl;
    });

    QUnit.test('castStreamUrl should be identical to streamUrl except without allow_source', function(assert) {
        const accessToken = this.api.expectVodAccessToken(this.vodId, {
            canAccess: true,
            accessTokenParams: this.accessTokenParams,
            oauthToken: this.oauthToken,
        });
        const oAuthResponse = Promise.resolve(this.oauthToken);

        const stream = new VODTwitchContentStream(
            this.vodId,
            oAuthResponse,
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );

        const params = {
            nauth: accessToken.token,
            nauthsig: accessToken.sig,
            godlike: true,
        };

        const streamUrl = stream.castStreamUrl.then(url => {
            assert.equal(
                url,
                // eslint-disable-next-line max-len
                `${USHER_HOST}/vod/${this.vodId.substring(1)}.m3u8?${stringify(params)}`
            );
        });

        return streamUrl;
    });

    QUnit.test('accessToken should be the correct value', function(assert) {
        const accessToken = this.api.expectVodAccessToken(this.vodId, {
            canAccess: true,
            accessTokenParams: this.accessTokenParams,
            oauthToken: this.oauthToken,
        });
        const stream = new VODTwitchContentStream(
            this.vodId,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams
        );

        return stream.accessToken.then(token => {
            assert.deepEqual(token, accessToken);
        });
    });

    QUnit.test('accessToken should return the same promise when called more than once before expiry', function(assert) {
        this.api.expectVodAccessToken(this.vodId, {
            canAccess: true,
            expires: Math.floor((Date.now() + 300000) / 1000),
            accessTokenParams: this.accessTokenParams,
            oauthToken: this.oauthToken,
        });
        const stream = new VODTwitchContentStream(
            this.vodId,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams
        );

        const firstPromise = stream.accessToken;
        const secondPromise = stream.accessToken;
        assert.equal(firstPromise, secondPromise);
    });

    QUnit.test('accessToken should return a new promise on expiry', function(assert) {
        const expires = Math.floor(Date.now() / 1000);
        const stream = new VODTwitchContentStream(
            this.vodId,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams
        );

        this.api.expectVodAccessToken(this.vodId, {
            canAccess: true,
            expires,
            accessTokenParams: this.accessTokenParams,
            oauthToken: this.oauthToken,
        });

        const firstPromise = stream.accessToken;
        return firstPromise.then(() => {
            const secondPromise = stream.accessToken;
            assert.notEqual(firstPromise, secondPromise);
        });
    });

    // eslint-disable-next-line max-len
    QUnit.test('accessToken should return a new promise when not expired and after resetNauthToken() is called', function(assert) {
        this.api.expectVodAccessToken(this.vodId, {
            canAccess: true,
            expires: Math.floor((Date.now() + 300000) / 1000),
            accessTokenParams: this.accessTokenParams,
            oauthToken: this.oauthToken,
        });
        const stream = new VODTwitchContentStream(
            this.vodId,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams
        );

        const firstPromise = stream.accessToken;
        stream.resetNAuthToken();
        const secondPromise = stream.accessToken;
        assert.notEqual(firstPromise, secondPromise);
    });
});
