import { TimelineMetadataManager } from 'timeline-metadata/manager';
import { apiHost } from 'settings';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { ACTION_SET_STREAM } from 'actions/stream';
import { waitFor } from 'tests/utils/waitFor';
import { unitTest } from 'tests/utils/module';
import { init as initStore } from 'state';
import { MARKERS } from 'experiments';
import { ACTION_SET_EXPERIMENTS } from 'actions/experiments';
import { TEST_VCA_MARKER_RESPONSE, TEST_NORMALIZED_VCA_MARKERS, TEST_VOD_MARKER_RESPONSE,
         TEST_NORMALIZED_VOD_MARKERS, TEST_MUTED_SEGMENTS_RESPONSE,
         TEST_NORMALIZED_MUTED_SEGMENTS } from 'tests/fixtures/timeline-metadata';

const SEEK_PREVIEW_URL = 'http://www.aseekpreviewurl.com/';
const VCA_BASE_MARKERS_URL = 'https://clipmn.twitch.tv/prod/';
const VCA_CHANNEL_AVAILABILITY_URL = 'https://clipmn.twitch.tv/prod/ids.json';
const VOD_BASE_MARKERS_URL = `${apiHost}/kraken/videos/`;

unitTest('timeline-metadata', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();

        this.createExperiment = options => {
            return {
                get: uuid => {
                    switch (uuid) {
                    case MARKERS:
                        return Promise.resolve(options.markers);
                    }
                },
            };
        };

        this.store.dispatch({
            type: ACTION_SET_EXPERIMENTS,
            experiments: this.createExperiment({
                markers: 'no',
            }),
        });
    });

    QUnit.module('on new VOD stream', function(hooks) {
        hooks.beforeEach(function() {
            this.videoID = 'v1234567';
            this.channelName = 'channel';
            this.channelID = 123456;

            this.api.setLoggedIn(false);

            this.api.expectVideoInfo(this.videoID, {
                channel: {
                    name: this.channelName,
                    _id: this.channelID,
                },

                // eslint-disable-next-line camelcase
                muted_segments: TEST_MUTED_SEGMENTS_RESPONSE,
                // eslint-disable-next-line camelcase
                seek_previews_url: SEEK_PREVIEW_URL,
            });

            this.api.expectChannelInfo(this.channelName, {
                _id: this.channelID,
            });
        });

        QUnit.test('sets mutedSegments in store to normalized mutedSegments', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const timelineMetadata = new TimelineMetadataManager(this.store);

            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(this.videoID),
            });

            return waitFor(() => this.store.getState().timelineMetadata.mutedSegments.length !== 0).then(() => {
                // eslint-disable-next-line max-len
                assert.deepEqual(this.store.getState().timelineMetadata.mutedSegments, TEST_NORMALIZED_MUTED_SEGMENTS);
            });
        });

        QUnit.test('sets previews in store to normalized previews', function(assert) {
            const seekPreviewResponse = [
                {
                    count: 199,
                    width: 110,
                    rows: 40,
                    images: ['110188780-low-0.jpg'],
                    interval: 113,
                    quality: 'low',
                    cols: 5,
                    height: 62,
                },
                {
                    count: 199,
                    width: 220,
                    rows: 10,
                    images: [
                        '110188780-high-0.jpg',
                        '110188780-high-1.jpg',
                        '110188780-high-2.jpg',
                        '110188780-high-3.jpg',
                    ],
                    interval: 113,
                    quality: 'high',
                    cols: 5,
                    height: 124,
                },
            ];

            const normalizedPreviews = {
                count: 199,
                hq: {
                    URLs: [
                        'http://www.aseekpreviewurl.com/110188780-high-0.jpg',
                        'http://www.aseekpreviewurl.com/110188780-high-1.jpg',
                        'http://www.aseekpreviewurl.com/110188780-high-2.jpg',
                        'http://www.aseekpreviewurl.com/110188780-high-3.jpg',
                    ],
                    cols: 5,
                    height: 124,
                    rows: 10,
                    width: 220,
                },
                lq: {
                    URLs: ['http://www.aseekpreviewurl.com/110188780-low-0.jpg'],
                    cols: 5,
                    height: 62,
                    rows: 40,
                    width: 110,
                },
            };

            this.api.mock(SEEK_PREVIEW_URL, { body: seekPreviewResponse });

            // eslint-disable-next-line no-unused-vars
            const timelineMetadata = new TimelineMetadataManager(this.store);

            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(this.videoID),
            });

            return waitFor(() => this.store.getState().timelineMetadata.previews.count !== 0).then(() => {
                assert.deepEqual(this.store.getState().timelineMetadata.previews, normalizedPreviews);
            });
        });

        QUnit.module('markers', function() {
            // eslint-disable-next-line max-len
            QUnit.test('sets markers in store to normalized VCA markers when MARKERS experiment is `vca`', function(assert) {
                this.api.mock(VCA_CHANNEL_AVAILABILITY_URL, { body: [this.channelID] });
                this.api.mock(`${VCA_BASE_MARKERS_URL}${this.videoID}.json`, { body: TEST_VCA_MARKER_RESPONSE });

                this.store.dispatch({
                    type: ACTION_SET_EXPERIMENTS,
                    experiments: this.createExperiment({
                        markers: 'vca',
                    }),
                });

                // eslint-disable-next-line no-unused-vars
                const timelineMetadata = new TimelineMetadataManager(this.store);

                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: new VODTwitchContentStream(this.videoID),
                });

                return waitFor(() => this.store.getState().timelineMetadata.markers.length !== 0).then(() => {
                    assert.deepEqual(this.store.getState().timelineMetadata.markers, TEST_NORMALIZED_VCA_MARKERS);
                });
            });

            // eslint-disable-next-line max-len
            QUnit.test('sets markers in store to normalized VOD markers when MARKERS experiment is `vod`', function(assert) {
                const trimmedVideoID = this.videoID.substring(this.videoID.indexOf('v') + 1);
                this.api.mock(`${VOD_BASE_MARKERS_URL}${trimmedVideoID}/markers`, { body: TEST_VOD_MARKER_RESPONSE });

                this.store.dispatch({
                    type: ACTION_SET_EXPERIMENTS,
                    experiments: this.createExperiment({
                        markers: 'vod',
                    }),
                });

                // eslint-disable-next-line no-unused-vars
                const timelineMetadata = new TimelineMetadataManager(this.store);

                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: new VODTwitchContentStream(this.videoID),
                });

                return waitFor(() => this.store.getState().timelineMetadata.markers.length !== 0).then(() => {
                    assert.deepEqual(this.store.getState().timelineMetadata.markers, TEST_NORMALIZED_VOD_MARKERS);
                });
            });
        });
    });
});
