import { init as initStore } from 'state';
import sinon from 'sinon';
import { TwitchEverywhereManager } from 'twitch-everywhere-manager';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { userLoggedIn } from 'actions/user';
import * as FollowActions from 'actions/follow';
import * as SubscriptionsActions from 'actions/subscriptions';

QUnit.module('twitch-everywhere-manager', function(hooks) {
    hooks.beforeEach(function() {
        this.sandbox = sinon.sandbox.create();

        this.sandbox.stub(FollowActions, 'fetchFollowInfo', (userId, channelId) => ({
            type: 'fake-fetch-follow-info',
            userId,
            channelId,
        }));

        this.sandbox.stub(SubscriptionsActions, 'fetchChannelSubscriptions', (userId, channelId) => ({
            type: 'fake-fetch-channel-subscriptions',
            userId,
            channelId,
        }));

        this.sandbox.stub(SubscriptionsActions, 'fetchSubscriptionProducts', channelName => ({
            type: 'fake-fetch-subscription-products',
            channelName,
        }));

        this.store = initStore();

        this.sandbox.spy(this.store, 'dispatch');

        this.twitchEverywhereManager = new TwitchEverywhereManager(this.store);
    });

    hooks.afterEach(function() {
        this.sandbox.restore();
    });

    QUnit.test('should fetch followInfo and subcriptions when the channel changes', function(assert) {
        const dispatch = this.store.dispatch;
        const userId = 321;
        const channelId = 123;
        const channelName = 'foo';

        dispatch(userLoggedIn({ id: userId }));

        dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                channel: {
                    id: channelId,
                    name: channelName,
                },
            },
        });

        assert.equal(dispatch.callCount, 5, 'dispatch called 5 times');

        assert.ok(dispatch.calledWith({
            type: 'fake-fetch-follow-info',
            userId: userId,
            channelId: channelId,
        }));

        assert.ok(dispatch.calledWith({
            type: 'fake-fetch-channel-subscriptions',
            userId: userId,
            channelId: channelId,
        }));

        assert.ok(dispatch.calledWith({
            type: 'fake-fetch-subscription-products',
            channelName: channelName,
        }));
    });

    QUnit.test('does fetch only subscriptionProducts when not logged in', function(assert) {
        const dispatch = this.store.dispatch;
        const channelName = 'foo';

        dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                channel: {
                    id: 123,
                    name: channelName,
                },
            },
        });

        assert.equal(dispatch.callCount, 2, 'dispatch called 2 times');

        assert.ok(dispatch.calledWith({
            type: 'fake-fetch-subscription-products',
            channelName: channelName,
        }));
    });
});
