import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { mount, shallow } from 'enzyme';
import { Button } from 'ui/components/buttons/base-button';

const TestComponent = () => <span />;

const DEFAULT_ARGS = Object.freeze({
    children: [<TestComponent key={0} />],
    className: 'test-base-button',
    disabled: false,
    id: 'uniqueness',
    onBlur() {},
    onClick: () => {},
    onFocus() {},
    onMouseOut() {},
    onMouseOver() {},
    tabIndex: '0',
});

function renderButton(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <Button {...args} />;
    return shallow(component);
}

function mountRenderButton(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <Button {...args} />;
    return mount(component);
}

reactTest('ui | components | buttons | base-button', function() {
    QUnit.test('renders a button to the DOM', function(assert) {
        const component = renderButton();
        assert.equal(component.type(), 'button');
    });

    QUnit.test('fires `onClick` when clicked', function(assert) {
        const onClick = sinon.spy();
        const component = renderButton({ onClick });
        component.simulate('click');
        assert.equal(onClick.callCount, 1);
    });

    QUnit.test('fires `onClick` when enter key is hit', function(assert) {
        const onClick = sinon.spy();
        const component = renderButton({ onClick });
        component.simulate('keyup', { key: 'Enter' });
        assert.equal(onClick.callCount, 1);
    });

    QUnit.test('renders child components', function(assert) {
        const component = renderButton();
        assert.ok(component.containsMatchingElement(
            <TestComponent />
        ));
    });

    QUnit.test('does not fire `onClick` when button is disabled', function(assert) {
        const onClick = sinon.spy();
        const component = mountRenderButton({
            disabled: true,
            onClick,
        });
        component.simulate('click');
        assert.equal(onClick.callCount, 0);
    });

    QUnit.test('has expected `tabIndex` value on button', function(assert) {
        const tabIndex = '5';
        const component = renderButton({
            tabIndex,
        });

        assert.equal(component.find('button').props().tabIndex, tabIndex);
    });

    QUnit.test('has expected `id` value on button', function(assert) {
        const id = 'newuniqueness';
        const component = renderButton({
            id,
        });

        assert.equal(component.find('button').props().id, id);
    });
});
