import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { IconComponent } from 'ui/components/buttons/base-icon';

const DEFAULT_ARGS = Object.freeze({
    labelClass: 'test-base-icon',
    label: 'test',
    iconId: 'id',
    t: str => str,
});

function renderIcon(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <IconComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | buttons | base-icon', function() {
    QUnit.test('renders a span to the DOM', function(assert) {
        const component = renderIcon();
        assert.equal(component.type(), 'span');
    });

    QUnit.test('label always has class `player-tip`', function(assert) {
        const component = renderIcon({
            labelClass: '',
        });
        const expectedClassName = 'player-tip';
        const { className: actualClassName } = component.childAt(0).props();
        assert.equal(expectedClassName, actualClassName);
    });

    QUnit.test('creates a label span with the supplied label', function(assert) {
        const label = QUnit.config.current.testId;
        const component = renderIcon({ label });
        const { 'data-tip': actualLabel } = component.childAt(0).props();
        assert.equal(actualLabel, label);
    });

    QUnit.test('includes label span when passed non-blank label string', function(assert) {
        const component = renderIcon();
        assert.equal(component.contains(<span className={'player-tip test-base-icon'} data-tip={'test'} />), true);
    });

    QUnit.test('omits label span with blank label string', function(assert) {
        const component = renderIcon({ label: '' });
        assert.equal(component.contains(<span className={'player-tip test-base-icon'} data-tip={''} />), false);
    });

    QUnit.test('sets the xlinkHref of the svg properly', function(assert) {
        const iconId = QUnit.config.current.testId;
        const component = renderIcon({ iconId });
        const { xlinkHref: actualIconId } = component.find('use').props();
        assert.equal(actualIconId, iconId);
    });
});
