import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ChromecastButton } from 'ui/components/buttons/chromecast-button';
import { AVAILABLE, CONNECTING, UNAVAILABLE,
         CONNECTED } from 'state/chromecast';
import { Button } from 'ui/components/buttons/base-button';
import { Icon } from 'ui/components/buttons/base-icon';

const DEFAULT_ARGS = Object.freeze({
    castingState: UNAVAILABLE,
    onClick() {},
});

function renderChromecastButton(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ChromecastButton {...args} />;
    return shallow(component);
}

reactTest('ui | components | buttons | chromecast-button', function() {
    QUnit.test('renders null if castingState is UNAVAILABLE', function(assert) {
        const component = renderChromecastButton();
        assert.equal(component.type(), null);
    });

    QUnit.test('renders null if castingState is invalid', function(assert) {
        const component = renderChromecastButton({
            castingState: QUnit.config.current.testId,
        });
        assert.equal(component.type(), null);
    });

    function testCastingState(castingState, className) {
        const component = renderChromecastButton({
            castingState,
        });

        QUnit.test(`if castingState is ${castingState}, expect className ${className}`, function(assert) {
            assert.ok(component.hasClass(className), `has class ${className}`);
        });

        QUnit.test('renders a Button', function(assert) {
            assert.equal(component.type(), Button, 'renders a Button');
        });

        QUnit.test('Has an Icon as a base svg', function(assert) {
            const iconComponent = component.childAt(0);
            const iconProps = iconComponent.props();

            assert.equal(iconComponent.type(), Icon);
            assert.deepEqual(iconProps, {
                label: 'Chromecast',
                iconId: '#icon_chromecast_p1',
            }, 'Correct props passed to Icon');
        });
    }

    testCastingState(AVAILABLE, 'pl-chromecast--available');
    testCastingState(CONNECTING, 'pl-chromecast--connecting');
    testCastingState(CONNECTED, 'pl-chromecast--connected');

    QUnit.test('invokes onClick handler when clicked', function(assert) {
        const onClick = sinon.spy();
        const component = renderChromecastButton({
            castingState: AVAILABLE,
            onClick,
        });

        component.simulate('click');

        assert.equal(onClick.callCount, 1);
    });
});
