import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { FullScreenButton } from 'ui/components/buttons/fullscreen-button';
import * as buttonTestHelper from 'tests/utils/player-buttons-helpers';

reactTest('ui | components | buttons | fullscreen-button', function(hooks) {
    hooks.beforeEach(function() {
        this.renderFullScreenButton = buttonTestHelper.createRenderFn(FullScreenButton, {
            canFullScreen: false,
            hasBranding: false,
            isFullScreen: false,
            onFullScreen: () => {},
            onUnFullScreen: () => {},
        });
    });

    QUnit.module('when isFullScreen', function() {
        runFullScreenSuite(true);
    });

    QUnit.module('when not isFullScreen', function() {
        runFullScreenSuite(false);
    });

    // eslint-disable-next-line complexity
    function runFullScreenSuite(isFullScreen) {
        const label = isFullScreen ?  'Exit Fullscreen' : 'Fullscreen';
        const iconId = isFullScreen ? '#icon_unfullscreen' : '#icon_fullscreen';
        const onFullScreen = sinon.spy();
        const onUnFullScreen = sinon.spy();
        const shouldFire = isFullScreen ? onUnFullScreen : onFullScreen;
        const shouldNotFire = isFullScreen ? onFullScreen : onUnFullScreen;

        QUnit.test(`label should be ${label}`, function(assert) {
            const component = this.renderFullScreenButton({ isFullScreen });
            buttonTestHelper.testLabel(component, label, assert);
        });

        QUnit.test(`should fire ${isFullScreen ? 'onFullScreen' : 'onUnFullScreen'} when clicked`, function(assert) {
            const component = this.renderFullScreenButton({
                isFullScreen,
                onFullScreen,
                onUnFullScreen,
            });

            buttonTestHelper.testClickHandler({
                component,
                shouldFire,
                shouldNotFire,
                assert,
            });
        });

        QUnit.test(`should have icon id ${iconId}`, function(assert) {
            const component = this.renderFullScreenButton({ isFullScreen });
            buttonTestHelper.testIconId(component, iconId, assert);
        });
    }

    QUnit.test('should have cannot-fullscreen class if canFullScreen is false', function(assert) {
        const component = this.renderFullScreenButton({
            canFullScreen: false,
        });
        buttonTestHelper.testButtonHasClass(component, 'pl-button__fullscreen--cannot-fullscreen', assert);
    });

    QUnit.test('should not have cannot-fullscreen class if canFullScreen is true', function(assert) {
        const component = this.renderFullScreenButton({
            canFullScreen: true,
        });
        buttonTestHelper.testButtonNotHasClass(component, 'pl-button__fullscreen--cannot-fullscreen', assert);
    });

    QUnit.test('should have margin right and tooltip left classes if branding is not shown', function(assert) {
        const component = this.renderFullScreenButton({
            hasBranding: false,
        });
        buttonTestHelper.testButtonHasClass(component, 'pl-mg-r-1', assert);
        buttonTestHelper.testButtonHasClass(component, 'pl-button__fullscreen--tooltip-left', assert);
    });

    QUnit.test('should not have margin right and tooltip left classes if branding is shown', function(assert) {
        const component = this.renderFullScreenButton({
            hasBranding: true,
        });
        buttonTestHelper.testButtonNotHasClass(component, 'pl-mg-r-1', assert);
        buttonTestHelper.testButtonNotHasClass(component, 'pl-button__fullscreen--tooltip-left', assert);
    });

    QUnit.test('when cannot fullscreen, button should be disabled', function(assert) {
        const component = this.renderFullScreenButton({
            canFullScreen: false,
        });
        buttonTestHelper.testButtonDisabled(component, assert);
    });

    QUnit.test('when can fullscreen, button should not be disabled', function(assert) {
        const component = this.renderFullScreenButton({
            canFullScreen: true,
        });
        buttonTestHelper.testButtonNotDisabled(component, assert);
    });
});
