import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { BGandWindowPanel } from 'ui/components/closed-captions/bg-and-window-panel';
import { ColorPalette } from 'ui/components/closed-captions/color-palette';
import { OpacityDropdown } from 'ui/components/closed-captions/opacity-dropdown';

const DEFAULT_ARGS = Object.freeze({
    currentColor: '',
    currentOpacity: '',
    selectColor() {},
    selectOpacity() {},
});

const COLORS = Object.freeze([
    'transparent',
    'white',
    'lightgray',
    'gray',
    'darkgray',
    'black',
    'green',
    'blue',
    'cyan',
    'magenta',
    'yellow',
    'red',
]);

function renderBGandWindowPanel(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <BGandWindowPanel {...args} />;
    return shallow(component);
}

reactTest('ui | components | closed-captions | bg-and-window-panel', function() {
    QUnit.test('renders a div with the correct className', function(assert) {
        const component = renderBGandWindowPanel();
        assert.equal(component.type(), 'div', 'is div');
        assert.ok(component.hasClass('cc-panel-menu-frame'), 'has right class');
    });

    QUnit.test('renders correct children', function(assert) {
        const component = renderBGandWindowPanel();
        assert.equal(
            component.childAt(0).type(),
            OpacityDropdown,
            'renders an opacity dropdown'
        );
        assert.equal(
            component.childAt(1).type(),
            ColorPalette,
            'renders a color palette'
        );
    });

    QUnit.test('correct props set on OpacityDropdown', function(assert) {
        const currentOpacity = QUnit.config.current.testId;
        const component = renderBGandWindowPanel({ currentOpacity });
        const dropdown = component.childAt(0);

        assert.equal(
            dropdown.props().currentOpacity,
            currentOpacity,
            'passes currentOpacity'
        );

        assert.deepEqual(
            dropdown.props().onSelect,
            DEFAULT_ARGS.selectOpacity,
            'correct method passed'
        );
    });

    QUnit.test('correct props set on ColorPalette', function(assert) {
        const currentColor = QUnit.config.current.testId;
        const component = renderBGandWindowPanel({ currentColor });
        const colorPalette = component.childAt(1);

        assert.equal(
            colorPalette.props().currentColor,
            currentColor,
            'passes currentColor'
        );

        assert.deepEqual(
            colorPalette.props().onSelect,
            DEFAULT_ARGS.selectColor,
            'correct method passed'
        );

        assert.deepEqual(
            colorPalette.props().colors,
            COLORS,
            'passes in correct colors'
        );
    });
});
