import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { CCPanelTabsComponent } from 'ui/components/closed-captions/closed-captions-panel-tabs';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';
import { PRESETS } from 'ui/containers/closed-captions/presets-panel';
import { TEXT } from 'ui/containers/closed-captions/text-panel';
import { EFFECTS } from 'ui/containers/closed-captions/effects-panel';
import { BACKGROUND } from 'ui/containers/closed-captions/background-panel';
import { WINDOW } from 'ui/containers/closed-captions/window-panel';

const DEFAULT_ARGS = Object.freeze({
    currentPanel: PRESETS,
    selectPanel: () => {},
    t: mockTranslateFunc,
});

function renderCCPanelTabs(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <CCPanelTabsComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | closed-captions | closed-captions-panel-tabs', function() {
    QUnit.test('should render ul with correct class names', function(assert) {
        const component = renderCCPanelTabs();
        assert.equal(component.type(), 'ul');
        assert.ok(component.hasClass('player-tabs'));
        assert.ok(component.hasClass('clearfix'));
    });

    QUnit.test('should have five li items with player-tabs__item class', function(assert) {
        const component = renderCCPanelTabs();
        assert.equal(component.find('li').length, 5);
        component.find('li').
            forEach(elm => {
                assert.ok(elm.hasClass('player-tabs__item'), 'has player-tabs__item class');
            });
    });

    function testTabs(currentPanel) {
        QUnit.test(`when currentPanel is ${currentPanel}, should add active class`, function(assert) {
            const component = renderCCPanelTabs({ currentPanel });
            component.find('li').
                forEach(elm => {
                    if (elm.text() !== currentPanel) {
                        return;
                    }
                    assert.ok(elm.hasClass('player-tabs__item--active'), `${currentPanel} is active`);
                });
        });

        QUnit.test('passes tab name to selectPanel when clicked', function(assert) {
            const selectPanel = sinon.spy();
            const component = renderCCPanelTabs({ selectPanel });
            component.
                findWhere(elm => {
                    return elm.type() === 'a' && elm.text() === currentPanel;
                }).
                simulate('click');

            assert.equal(selectPanel.callCount, 1, 'called once');
            assert.equal(selectPanel.firstCall.args[0], currentPanel);
        });
    }

    testTabs(PRESETS);
    testTabs(TEXT);
    testTabs(EFFECTS);
    testTabs(BACKGROUND);
    testTabs(WINDOW);
});
