import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ClosedCaptionsPanelComponent } from 'ui/components/closed-captions/closed-captions-panel';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';
import { CCPanelTabs } from 'ui/components/closed-captions/closed-captions-panel-tabs';
import { PresetsPanel, PRESETS } from 'ui/containers/closed-captions/presets-panel';
import { TextPanel, TEXT } from 'ui/containers/closed-captions/text-panel';
import { EffectsPanel, EFFECTS } from 'ui/containers/closed-captions/effects-panel';
import { BackgroundPanel, BACKGROUND } from 'ui/containers/closed-captions/background-panel';
import { WindowPanel, WINDOW } from 'ui/containers/closed-captions/window-panel';

const DEFAULT_ARGS = Object.freeze({
    onClose: () => {},
    t: mockTranslateFunc,
});

function renderCCPanel(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ClosedCaptionsPanelComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | closed-captions | closed-captions-modal', function() {
    QUnit.test('renders a div with the correct class', function(assert) {
        const component = renderCCPanel();
        assert.equal(component.type(), 'div', 'is div');
        assert.ok(component.hasClass('pl-modal__content'));
    });

    QUnit.test('has a close button that invokes onClose when clicked', function(assert) {
        const onClose = sinon.spy();
        const component = renderCCPanel({ onClose });
        const closeButton = component.find('.pl-modal__close');

        assert.equal(closeButton.length, 1, 'one close button');

        closeButton.simulate('click');

        assert.equal(onClose.callCount, 1, 'onClose called once');
    });

    QUnit.test('has a title', function(assert) {
        const component = renderCCPanel();
        const title = component.find('.pl-modal__header');
        assert.equal(title.text(), 'Closed Caption Settings', 'correct title rendered');
    });

    function generateContentTest(currentPanel, modal) {
        QUnit.test(`has a CCPanelTabs component for ${currentPanel} with correct props`, function(assert) {
            const component = renderCCPanel();
            const selectPanel = component.instance().selectPanel;
            selectPanel(currentPanel);

            component.update();

            const tabs = component.find(CCPanelTabs);

            assert.equal(tabs.length, 1, 'one CCPanelTabs element');

            assert.deepEqual(tabs.props(), {
                selectPanel,
                currentPanel,
            }, `when modal is ${currentPanel}, pass it to tabs component`);
        });

        QUnit.test(`generates the correct component when currentPanel changes to ${currentPanel}`, function(assert) {
            const component = renderCCPanel();
            const selectPanel = component.instance().selectPanel;
            selectPanel(currentPanel);

            component.update();

            const content = component.find(modal);

            assert.equal(content.length, 1, `contains 1 ${currentPanel} element`);
        });
    }

    generateContentTest(PRESETS, PresetsPanel);
    generateContentTest(TEXT, TextPanel);
    generateContentTest(EFFECTS, EffectsPanel);
    generateContentTest(BACKGROUND, BackgroundPanel);
    generateContentTest(WINDOW, WindowPanel);
});
