import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ColorPaletteComponent } from 'ui/components/closed-captions/color-palette';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const COLORS = Object.freeze([
    'transparent',
    'white',
    'lightgray',
    'gray',
    'darkgray',
    'black',
    'green',
    'blue',
    'cyan',
    'magenta',
    'yellow',
    'red',
]);

const DEFAULT_ARGS = Object.freeze({
    currentColor: '',
    colors: COLORS,
    onSelect() {},
    t: mockTranslateFunc,
});

function renderColorPalette(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ColorPaletteComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | closed-captions | color-palette', function() {
    QUnit.test('renders a div with the correct classes', function(assert) {
        const component = renderColorPalette();
        assert.equal(component.type(), 'div', 'renders a div');
        assert.ok(component.hasClass('cc-customize-field'), 'has cc panel container class');
    });

    QUnit.test('renders correct text', function(assert) {
        const component = renderColorPalette();
        assert.equal(component.text(), 'Color', 'renders correct label');
    });

    QUnit.test('when color is transparent, renders a diagonal line', function(assert) {
        const component = renderColorPalette({
            currentColor: 'transparent',
        });
        const line = component.find('.cc-no-color');
        assert.equal(line.type(), 'hr', 'renders an hr element');
    });

    function testColorSquares(currentColor) {
        QUnit.test(`when color is ${currentColor} has correct selected class`, function(assert) {
            const component = renderColorPalette({ currentColor });
            const selectedColorComponent = component.find('.cc-color-palette__square-selected');
            assert.ok(
                selectedColorComponent.hasClass(`cc-color-palette__square-${currentColor}`),
                `has ${currentColor} class`
            );
        });

        QUnit.test(`when ${currentColor} square is clicked passes ${currentColor} to onSelect`, function(assert) {
            const onSelect = sinon.spy();
            const component = renderColorPalette({
                onSelect,
                currentColor,
            });
            component.find('.cc-color-palette__square-selected').simulate('click');

            assert.equal(onSelect.callCount, 1, 'called once');
            assert.equal(onSelect.firstCall.args[0], currentColor, 'passes current color');
        });

        QUnit.test(`onKeyUp ${currentColor} is passed to onSelect`, function(assert) {
            const onSelect = sinon.spy();
            const component = renderColorPalette({
                onSelect,
                currentColor,
            });
            component.find('.cc-color-palette__square-selected').simulate('keyup', { key: 'Enter' });

            assert.equal(onSelect.callCount, 1, 'called once');
            assert.equal(onSelect.firstCall.args[0], currentColor, 'passes current color');
        });
    }

    COLORS.forEach(testColorSquares);
});
