import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PresetsPanel, CC_PRESET_VALUES, isTransparentBG } from 'ui/components/closed-captions/presets-panel';

const DEFAULT_ARGS = Object.freeze({
    onSelect() {},
    selectedValue: CC_PRESET_VALUES[0],
});

const TRANSPARENT_VALUES = Object.freeze([
    'white-on-trans-black',
    'lime-on-trans-black',
    'yellow-on-trans-black',
    'cyan-on-trans-black',
    'magenta-on-trans-black',
]);

function renderPresetsPanel(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <PresetsPanel {...args} />;
    return shallow(component);
}

reactTest('ui | components | closed-captions | presets-panel', function() {
    QUnit.test('should render a div with correct classes', function(assert) {
        const component = renderPresetsPanel();
        assert.equal(component.type(), 'div');
        assert.ok(component.hasClass('cc-panel-menu-frame'), 'has base class');
        assert.ok(component.hasClass('cc-panel-menu-frame__presets'), 'has presets class');
    });

    QUnit.test('isTransparentBG should catch preset values that are transparent', function(assert) {
        CC_PRESET_VALUES.
            filter(isTransparentBG).
            forEach(presetValue => {
                assert.ok(
                    TRANSPARENT_VALUES.indexOf(presetValue) > -1,
                    'catches all transparent values'
                );
            });
    });

    QUnit.test('11 presets are shown in menu', function(assert) {
        const component = renderPresetsPanel();
        const presets = component.find('.cc-preset-square');

        assert.equal(presets.length, 11, '11 presets');

        presets.forEach(preset => {
            assert.equal(preset.type(), 'div', 'renders a div');
            assert.equal(preset.text(), 'Aa', 'renders correct text');
        });
    });

    function testPresetButton(presetValue) {
        QUnit.test(`should pass ${presetValue} to onSelect when clicked`, function(assert) {
            const onSelect = sinon.spy();
            const component = renderPresetsPanel({ onSelect });
            component.
                find(`.cc-preset-square[data-preset="${presetValue}"]`).
                simulate('click');

            assert.equal(onSelect.callCount, 1, 'called once');
            assert.equal(onSelect.firstCall.args[0], presetValue, `called with ${presetValue}`);
        });

        QUnit.test(`should pass ${presetValue} to onSelect when enter key is hit`, function(assert) {
            const onSelect = sinon.spy();
            const component = renderPresetsPanel({ onSelect });
            component.
                find(`.cc-preset-square[data-preset="${presetValue}"]`).
                simulate('keyup', { key: 'Enter' });

            assert.equal(onSelect.callCount, 1, 'called once');
            assert.equal(onSelect.firstCall.args[0], presetValue, `called with ${presetValue}`);
        });

        QUnit.test('adds js-cc-preset-selected to selected preset', function(assert) {
            const component = renderPresetsPanel({ selectedValue: presetValue });
            const preset = component.find(`.cc-preset-square[data-preset="${presetValue}"]`);

            assert.ok(
                preset.hasClass('js-cc-preset-selected'),
                'has selected class'
            );
        });
    }

    CC_PRESET_VALUES.forEach(testPresetButton);
});
