import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { TextEffectsPaletteComponent, NONE, RAISED,
         DROP, DEPRESSED, UNIFORM } from 'ui/components/closed-captions/text-effects-palette';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    currentTextEffect: NONE,
    selectTextEffect() {},
    t: mockTranslateFunc,
});

function renderTextEffectsPaletteComponent(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <TextEffectsPaletteComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | closed-captions | text-effects-palette', function() {
    QUnit.test('renders a div with the correct class', function(assert) {
        const component = renderTextEffectsPaletteComponent();
        assert.equal(component.type(), 'div');
        assert.ok(component.hasClass('cc-customize-field'), 'has correct class');
    });

    QUnit.test('renders correct copy', function(assert) {
        const component = renderTextEffectsPaletteComponent();
        const label = component.childAt(0);
        assert.equal(label.text(), 'Effect', 'renders correct copy');
    });

    function testButton(textEffect) {
        QUnit.test(`a cc-edge-palette__square--${textEffect} component exists`, function(assert) {
            const component = renderTextEffectsPaletteComponent();
            assert.equal(
                component.find(`.cc-edge-palette__square--${textEffect}`).length,
                1,
                `has one ${textEffect} button`
            );
        });

        QUnit.test(`adds a selected class if ${textEffect} is selected`, function(assert) {
            const component = renderTextEffectsPaletteComponent({
                currentTextEffect: textEffect,
            });
            const button = component.find(`.cc-edge-palette__square--${textEffect}`);

            assert.ok(button.hasClass('pl-cc-button--selected'), 'has selected class');
        });

        QUnit.test(`passes ${textEffect} to selectTextEffect onClick and onKeyUp`, function(assert) {
            const selectTextEffect = sinon.spy();
            const component = renderTextEffectsPaletteComponent({
                selectTextEffect,
            });
            const button = component.find(`.cc-edge-palette__square--${textEffect}`);

            button.simulate('click');

            assert.equal(selectTextEffect.callCount, 1, 'called once on click');
            assert.equal(
                selectTextEffect.firstCall.args[0],
                textEffect,
                `passes ${textEffect} to method`
            );

            selectTextEffect.reset();

            button.simulate('keyup', { key: 'Enter' });

            assert.equal(selectTextEffect.callCount, 1, 'called once on key up');
            assert.equal(
                selectTextEffect.firstCall.args[0],
                textEffect,
                `passes ${textEffect} to method`
            );
        });
    }

    testButton(NONE);
    testButton(RAISED);
    testButton(DROP);
    testButton(DEPRESSED);
    testButton(UNIFORM);
});
