import React from 'react';
import sinon from 'sinon';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { fontSizeMap } from 'captions/captions-style-map';
import { TextPanel } from 'ui/components/closed-captions/text-panel';
import { ColorPalette } from 'ui/components/closed-captions/color-palette';
import { FontDropdown } from 'ui/components/closed-captions/font-dropdown';
import { AlignmentDropdown } from 'ui/components/closed-captions/alignment-dropdown';
import { PositionDropdown } from 'ui/components/closed-captions/position-dropdown';
import { FontOpacityDropdown } from 'ui/components/closed-captions/font-opacity-dropdown';
import { SizePalette } from 'ui/components/closed-captions/size-palette';

const DEFAULT_ARGS = Object.freeze({
    currentAlignment: '',
    currentColor: '',
    currentFont: '',
    currentPosition: '',
    currentFontSize: 20,
    currentOpacity: '',
    selectFontChange() {},
});

const COLORS = Object.freeze([
    'white',
    'black',
    'red',
    'green',
    'cyan',
    'magenta',
    'yellow',
    'blue',
]);

function renderTextPanel(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <TextPanel {...args} />;
    return shallow(component);
}

reactTest('ui | components | closed-captions | text-panel', function() {
    QUnit.test('renders a div with the correct className', function(assert) {
        const component = renderTextPanel();
        assert.equal(component.type(), 'div', 'is div');
        assert.ok(component.hasClass('cc-panel-menu-frame'), 'has right class');
    });

    QUnit.test('renders correct children', function(assert) {
        const component = renderTextPanel();
        assert.equal(
            component.childAt(0).type(),
            FontDropdown,
            'renders an opacity dropdown'
        );
        assert.equal(
            component.childAt(1).type(),
            AlignmentDropdown,
            'renders an alignment dropdown'
        );
        assert.equal(
            component.childAt(2).type(),
            ColorPalette,
            'renders a color palette'
        );
        assert.equal(
            component.childAt(3).type(),
            PositionDropdown,
            'renders a position dropdown'
        );
        assert.equal(
            component.childAt(4).type(),
            FontOpacityDropdown,
            'renders a font opacity dropdown'
        );
        assert.equal(
            component.childAt(5).type(),
            SizePalette,
            'renders a size palette'
        );
    });

    function testDropdown(Dropdown, currentFontProp, selectFontProp, currentDropdown) {
        QUnit.test(`correct currentValue passed to ${currentDropdown}`, function(assert) {
            const currentValue = QUnit.config.current.testId;
            const component = renderTextPanel({
                [currentFontProp]: currentValue,
            });
            const dropdown = component.find(Dropdown);

            assert.equal(
                dropdown.props()[currentFontProp],
                currentValue,
                'passes currentValue'
            );
        });

        QUnit.test(`onSelect passes ${selectFontProp} to selectFontChange`, function(assert) {
            const selectFontChange = sinon.spy();

            const component = renderTextPanel({
                selectFontChange,
            });
            const dropdown = component.find(Dropdown);

            const value = QUnit.config.current.testId;

            dropdown.props().onSelect(value);

            assert.equal(
                selectFontChange.callCount,
                1,
                'selectFontChange called once'
            );
            assert.equal(
                selectFontChange.firstCall.args[0],
                selectFontProp,
                'called with correct fontProp'
            );
            assert.equal(
                selectFontChange.firstCall.args[1],
                value,
                'correct value passed'
            );
        });
    }

    testDropdown(FontDropdown, 'currentFont', 'font', 'FontDropdown');
    testDropdown(AlignmentDropdown, 'currentAlignment', 'alignment', 'AlignmentDropdown');
    testDropdown(PositionDropdown, 'currentPosition', 'verticalPosition', 'PositionDropdown');
    testDropdown(FontOpacityDropdown, 'currentOpacity', 'fontOpacity', 'FontOpacityDropdown');

    QUnit.test('correct props set on ColorPalette', function(assert) {
        const selectFontChange = sinon.spy();
        const currentColor = QUnit.config.current.testId;
        const component = renderTextPanel({
            currentColor,
            selectFontChange,
        });
        const colorPalette = component.find(ColorPalette);
        const selectedColor = 'red';

        assert.equal(
            colorPalette.props().currentColor,
            currentColor,
            'passes currentColor'
        );

        assert.deepEqual(
            colorPalette.props().colors,
            COLORS,
            'passes in correct colors'
        );

        colorPalette.props().onSelect(selectedColor);
        assert.equal(selectFontChange.callCount, 1, 'called once');
        assert.equal(selectFontChange.firstCall.args[0], 'fontColorName', 'correct font prop passed');
        assert.equal(selectFontChange.firstCall.args[1], selectedColor, 'correct value passed');
    });

    QUnit.test('correct methods set on SizePalette', function(assert) {
        const selectFontChange = sinon.spy();
        const component = renderTextPanel({
            selectFontChange,
        });
        const sizePalette = component.find(SizePalette);

        sizePalette.props().increaseSize();

        assert.equal(selectFontChange.callCount, 1, 'increaseSize called once');
        assert.equal(selectFontChange.firstCall.args[0], 'fontSize', 'fontSize prop selected');
        assert.equal(
            selectFontChange.firstCall.args[1],
            DEFAULT_ARGS.currentFontSize + fontSizeMap.increment,
            'increments the currentFontSize'
        );

        selectFontChange.reset();

        sizePalette.props().decreaseSize();

        assert.equal(selectFontChange.callCount, 1, 'decreaseSize called once');
        assert.equal(selectFontChange.firstCall.args[0], 'fontSize', 'fontSize prop selected');
        assert.equal(
            selectFontChange.firstCall.args[1],
            DEFAULT_ARGS.currentFontSize + fontSizeMap.decrement,
            'decrements the currentFontSize'
        );
    });
});
