import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { TextStylePaletteComponent } from 'ui/components/closed-captions/text-style-palette';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    boldText: false,
    italicText: false,
    underlineText: false,
    selectTextStyle() {},
    t: mockTranslateFunc,
});

function renderTextStylePaletteComponent(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <TextStylePaletteComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | closed-captions | text-style-palette', function() {
    QUnit.test('renders a div with the correct class', function(assert) {
        const component = renderTextStylePaletteComponent();
        assert.equal(component.type(), 'div');
        assert.ok(component.hasClass('cc-customize-field'), 'has correct class');
    });

    QUnit.test('renders correct copy', function(assert) {
        const component = renderTextStylePaletteComponent();
        const label = component.childAt(0);
        assert.equal(label.text(), 'Style', 'renders correct copy');
    });

    function testButton(fontProp, fontStyle, enabledProp) {
        QUnit.test(`a cc-style-palette__square--${fontStyle} component exists`, function(assert) {
            const component = renderTextStylePaletteComponent();
            assert.equal(
                component.find(`.cc-style-palette__square--${fontStyle}`).length,
                1,
                `has one ${fontStyle} button`
            );
        });

        QUnit.test(`adds a selected class if ${fontStyle} is selected`, function(assert) {
            const component = renderTextStylePaletteComponent({
                [`${fontStyle}Text`]: true,
            });
            const button = component.find(`.cc-style-palette__square--${fontStyle}`);

            assert.ok(button.hasClass('pl-cc-button--selected'), 'has selected class');
        });

        QUnit.test(`passes ${fontStyle} to selectTextStyle onClick and onKeyUp`, function(assert) {
            const selectTextStyle = sinon.spy();
            const component = renderTextStylePaletteComponent({
                selectTextStyle,
            });
            const button = component.find(`.cc-style-palette__square--${fontStyle}`);

            button.simulate('click');

            assert.equal(selectTextStyle.callCount, 1, 'called once on click');
            assert.equal(
                selectTextStyle.firstCall.args[0],
                fontProp,
                `sets fontStyle for ${fontProp}`
            );
            assert.equal(
                selectTextStyle.firstCall.args[1],
                fontStyle,
                `passes ${fontStyle} to method`
            );

            selectTextStyle.reset();

            button.simulate('keyup', { key: 'Enter' });

            assert.equal(selectTextStyle.callCount, 1, 'called once on key up');
            assert.equal(
                selectTextStyle.firstCall.args[0],
                fontProp,
                `sets fontStyle for ${fontProp}`
            );
            assert.equal(
                selectTextStyle.firstCall.args[1],
                fontStyle,
                `passes ${fontStyle} to method`
            );
        });

        QUnit.test(`passes none to selectTextStyle if ${enabledProp} is true`, function(assert) {
            const selectTextStyle = sinon.spy();
            const component = renderTextStylePaletteComponent({
                selectTextStyle,
                [enabledProp]: true,
            });
            const button = component.find(`.cc-style-palette__square--${fontStyle}`);
            button.simulate('click');

            assert.equal(selectTextStyle.callCount, 1, 'called once on click');
            assert.equal(
                selectTextStyle.firstCall.args[0],
                fontProp,
                `sets fontStyle for ${fontProp}`
            );
            assert.equal(
                selectTextStyle.firstCall.args[1],
                'none',
                'passes none to method'
            );
        });
    }

    testButton('fontUnderline', 'underline', 'underlineText');
    testButton('fontItalic', 'italic', 'italicText');
    testButton('fontBold', 'bold', 'boldText');
});
