import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { CollectionSidebarHeaderComponent } from 'ui/components/collections/collection-sidebar-header';
import { I18N_INSTANCE } from 'lang/i18n';
import { Interpolate } from 'react-i18next';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';
import { toHumanReadable } from 'util/duration-utils';
import { Button } from 'ui/components/buttons/base-button';

const DEFAULT_PROPS = Object.freeze({
    title: 'a title',
    ownerURL: 'a url',
    ownerName: 'a ownerName',
    itemsLength: 10,
    totalDuration: 123456,
    onClose: () => {},
    lang: {},
    t: mockTranslateFunc,
    i18n: I18N_INSTANCE, // needed for <Interpolate> component
});

function renderSidebarHeader(overrides = {}) {
    const props = assign({}, DEFAULT_PROPS, overrides);
    const component = <CollectionSidebarHeaderComponent {...props} />;
    return shallow(shallow(component).getElement());
}

reactTest('ui | components | collections | collection-sidebar-header', function() {
    QUnit.test('has correct top level classes', function(assert) {
        const sidebarHeaderComponent = renderSidebarHeader();
        const headerLayoutComponent = sidebarHeaderComponent.childAt(0);
        assert.ok(sidebarHeaderComponent.hasClass('collection-sidebar-header'));
        assert.ok(headerLayoutComponent.hasClass('collection-sidebar-header__layout'));
    });
    QUnit.test('has correct title element', function(assert) {
        const sidebarHeaderComponent = renderSidebarHeader();
        const titleElement = sidebarHeaderComponent.childAt(0).childAt(0);
        assert.ok(titleElement.hasClass('collection-sidebar-header__title'));
        assert.ok(titleElement.hasClass('qa-collection-header-title'));
        assert.ok(titleElement.hasClass('ellipsis'));
        assert.equal(titleElement.text(), DEFAULT_PROPS.title);
    });

    QUnit.test('has correct close button', function(assert) {
        const sidebarHeaderComponent = renderSidebarHeader();
        const closeButtonElement = sidebarHeaderComponent.childAt(1);
        assert.equal(closeButtonElement.type(), Button);
        assert.ok(closeButtonElement.hasClass('player-button'));
        assert.ok(closeButtonElement.childAt(0).containsMatchingElement(
            <svg><use xlinkHref="#icon_close_md" /></svg>
        ));
        assert.equal(closeButtonElement.prop('onClick'), DEFAULT_PROPS.onClose);
    });

    QUnit.test('has correct meta class', function(assert) {
        const sidebarHeaderComponent = renderSidebarHeader();
        const sidebarMetaElement = sidebarHeaderComponent.childAt(0).childAt(1);

        assert.ok(sidebarMetaElement.hasClass('collection-sidebar-header-meta'));
    });

    QUnit.test('has correctly translated meta author link element', function(assert) {
        const sidebarHeaderComponent = renderSidebarHeader();
        const translatedAuthorLinkElement = sidebarHeaderComponent.childAt(0).childAt(1).childAt(0);
        const expectedAuthorLinkElement = (
            <a
                className="qa-collection-sidebar-author"
                href={DEFAULT_PROPS.ownerURL}
            >
                {DEFAULT_PROPS.ownerName}
            </a>
        );

        assert.ok(translatedAuthorLinkElement.containsMatchingElement(
            <span>
                <Interpolate
                    i18nKey={'By {{author}}'}
                    author={expectedAuthorLinkElement}
                />
            </span>
        ));
    });

    QUnit.test('has correct meta item length element', function(assert) {
        const sidebarHeaderComponent = renderSidebarHeader();
        const dotDividerElement = sidebarHeaderComponent.childAt(0).childAt(1).childAt(1);
        const itemLengthElement = sidebarHeaderComponent.childAt(0).childAt(1).childAt(2);

        assert.ok(dotDividerElement.hasClass('middot-divider'));
        assert.equal(dotDividerElement.type(), 'span');
        assert.equal(itemLengthElement.type(), 'span');
        assert.equal(itemLengthElement.text(), `${DEFAULT_PROPS.itemsLength} video`);
    });

    QUnit.test('has correct classes and type on meta total duration element ', function(assert) {
        const sidebarHeaderComponent = renderSidebarHeader();
        const dotDividerElement = sidebarHeaderComponent.childAt(0).childAt(1).childAt(3);
        const totalDurationElement = sidebarHeaderComponent.childAt(0).childAt(1).childAt(4);

        assert.ok(dotDividerElement.hasClass('middot-divider'));
        assert.equal(dotDividerElement.type(), 'span');
        assert.equal(totalDurationElement.type(), 'span');
    });

    QUnit.test('has correct total duration text when there is only hours', function(assert) {
        const hoursOnlyDuration = 7200; // 2 hours
        const sidebarHeaderComponent = renderSidebarHeader({ totalDuration: hoursOnlyDuration });
        const totalDurationElement = sidebarHeaderComponent.childAt(0).childAt(1).childAt(4);

        const expectedTime = toHumanReadable(hoursOnlyDuration * 1000).hours;

        assert.equal(totalDurationElement.text(), `${expectedTime} hour`);
    });

    QUnit.test('has correct total duration text when there is only minutes', function(assert) {
        const minutesOnlyDuration = 123; // 2 minutes and 3 seconds
        const sidebarHeaderComponent = renderSidebarHeader({ totalDuration: minutesOnlyDuration });
        const totalDurationElement = sidebarHeaderComponent.childAt(0).childAt(1).childAt(4);

        const expectedTime = toHumanReadable(minutesOnlyDuration * 1000).minutes;

        assert.equal(totalDurationElement.text(), `${expectedTime} minute`);
    });

    QUnit.test('has correct total duration text when there is hours and minutes', function(assert) {
        const hoursAndMinutesDuration = 7400; // 2 hours and 3 minute and 20 seconds
        const sidebarHeaderComponent = renderSidebarHeader({ totalDuration: hoursAndMinutesDuration });
        const totalDurationElement = sidebarHeaderComponent.childAt(0).childAt(1).childAt(4);

        const expectedHourTime = toHumanReadable(hoursAndMinutesDuration * 1000).hours;
        const expectedMinuteTime = toHumanReadable(hoursAndMinutesDuration * 1000).minutes;

        assert.equal(totalDurationElement.text(), `${expectedHourTime} hour and ${expectedMinuteTime} minute`);
    });
});
