import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { CollectionSidebarItem } from 'ui/components/collections/collection-sidebar-item';
import sinon from 'sinon';
import assign from 'lodash/assign';

const DEFAULT_PROPS = {
    index: 0,
    onSelect: () => {},
    selectItem: 'select Item',
    isActive: false,
    thumbnailURL: 'a url',
    title: 'a title',
    subTitle: 'a subtitle',
};

function renderSidebarItem(propOverrides = {}) {
    const props = assign({}, DEFAULT_PROPS, propOverrides);
    const component = <CollectionSidebarItem {...props} />;
    return shallow(component);
}

reactTest('ui | components | collection | collection-sidebar-item', function() {
    QUnit.test('has correct classes', function(assert) {
        const sideBarItemComponent = renderSidebarItem();
        assert.ok(sideBarItemComponent.hasClass('collection-sidebar-item'));
        assert.ok(!sideBarItemComponent.hasClass('collection-sidebar-item--active'));
        assert.ok(sideBarItemComponent.hasClass('pl-flex'));
        assert.ok(sideBarItemComponent.hasClass('pl-flex--verticalCenter'));
        assert.ok(sideBarItemComponent.hasClass('pl-flex--nowrap'));
    });

    QUnit.test('adds active class when isActive is true', function(assert) {
        const sideBarItemComponent = renderSidebarItem({ isActive: true });
        assert.ok(sideBarItemComponent.hasClass('collection-sidebar-item--active'));
    });

    QUnit.test('has tabIndex and role prop', function(assert) {
        const sideBarItemComponent = renderSidebarItem();
        assert.equal(sideBarItemComponent.prop('role'), 'button');
        assert.equal(sideBarItemComponent.prop('tabIndex'), 0);
    });

    QUnit.test('has correct item number element', function(assert) {
        const sideBarItemComponent = renderSidebarItem();
        const itemNumberElement = sideBarItemComponent.childAt(0);
        assert.equal(itemNumberElement.text(), `${DEFAULT_PROPS.index + 1}`);
        assert.equal(itemNumberElement.type(), 'div');
        assert.ok(itemNumberElement.hasClass('collection-sidebar-item__number'));
        assert.ok(itemNumberElement.hasClass('pl-flex__item'));
        assert.ok(itemNumberElement.hasClass('qa-collection-item-number'));
    });

    QUnit.test('has correct thumbnail image', function(assert) {
        const sideBarItemComponent = renderSidebarItem();
        const thumbnailElement = sideBarItemComponent.childAt(1);
        assert.equal(thumbnailElement.type(), 'img');
        assert.ok(thumbnailElement.hasClass('collection-sidebar-item__thumbnail'));
        assert.ok(thumbnailElement.hasClass('pl-flex__item'));
        assert.ok(thumbnailElement.hasClass('qa-collection-item-thumbnail'));
        assert.equal(thumbnailElement.prop('src'), DEFAULT_PROPS.thumbnailURL);
        assert.equal(thumbnailElement.prop('alt'), 'Thumbnail of one of the videos in the collection.');
    });

    QUnit.test('has correct detail div', function(assert) {
        const sideBarItemComponent = renderSidebarItem();
        const detailElement = sideBarItemComponent.childAt(2);

        assert.ok(detailElement.hasClass('pl-flex__item'));
        assert.ok(detailElement.hasClass('collection-sidebar-item-detail'));
    });

    QUnit.test('has correct title div', function(assert) {
        const sideBarItemComponent = renderSidebarItem();
        const titleElement = sideBarItemComponent.childAt(2).childAt(0);

        assert.ok(titleElement.hasClass('collection-sidebar-item-detail__title'));
        assert.ok(titleElement.hasClass('qa-collection-item-title'));
        assert.equal(titleElement.text(), DEFAULT_PROPS.title);
    });

    QUnit.test('has correct subtitle span', function(assert) {
        const sideBarItemComponent = renderSidebarItem();
        const detailMetaElement = sideBarItemComponent.childAt(2).childAt(1);
        const subTitleElement = detailMetaElement.childAt(0);

        assert.ok(detailMetaElement.hasClass('collection-sidebar-item-detail__meta'));
        assert.ok(subTitleElement.hasClass('qa-collection-item-channel'));
        assert.equal(subTitleElement.text(), DEFAULT_PROPS.subTitle);
        assert.equal(subTitleElement.type(), 'span');
    });

    QUnit.test('attaches onClick handler that calls onSelect and passes selectItem in', function(assert) {
        const selectHandler = sinon.spy();
        const selectItem = {
            item: 'someItem',
        };

        const sideBarItemComponent = renderSidebarItem({
            onSelect: selectHandler,
            selectItem,
        });

        sideBarItemComponent.simulate('click');
        assert.equal(selectHandler.callCount, 1);
        assert.equal(selectHandler.firstCall.args[0], selectItem);
    });

    QUnit.test('attaches onKeyUp `Enter` handler that calls onSelect and passes selectItem in', function(assert) {
        const selectHandler = sinon.spy();
        const selectItem = {
            item: 'someItem',
        };

        const sideBarItemComponent = renderSidebarItem({
            onSelect: selectHandler,
            selectItem,
        });

        sideBarItemComponent.simulate('keyup', { key: 'Shift' });
        assert.equal(selectHandler.callCount, 0);
        sideBarItemComponent.simulate('keyup', { key: 'Enter' });
        assert.equal(selectHandler.callCount, 1);
        assert.equal(selectHandler.firstCall.args[0], selectItem);
    });
});
