import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { CollectionSidebarComponent, AnimatedCollectionSidebar } from 'ui/components/collections/collection-sidebar';
import { CollectionSidebarHeader } from 'ui/components/collections/collection-sidebar-header';
import { CollectionSidebarItem } from 'ui/components/collections/collection-sidebar-item';
import { TEST_NORMALIZED_COLLECTION } from 'tests/fixtures/collection';
import assign from 'lodash/assign';

const DEFAULT_PROPS = Object.freeze({
    currentItemIndex: 0,
    collection: TEST_NORMALIZED_COLLECTION,
    onSelectItem: () => {},
    onClose: () => {},
    className: '',
    onMouseEnter: () => {},
    onMouseLeave: () => {},
});

function renderSidebar(propOverrides = {}) {
    const props = assign({}, DEFAULT_PROPS, propOverrides);
    const component = <CollectionSidebarComponent {...props} />;
    return shallow(component);
}

reactTest('ui | components | collection | collection-sidebar', function() {
    QUnit.test('has correct class on top level', function(assert) {
        const sidebarComponent = renderSidebar();
        assert.ok(sidebarComponent.hasClass('collection-sidebar'));
    });

    QUnit.test('adds class from className prop to top level div', function(assert) {
        const expectedClass = 'an expected class';
        const sidebarComponent = renderSidebar({ className: expectedClass });
        assert.ok(sidebarComponent.hasClass('collection-sidebar'));
        assert.ok(sidebarComponent.hasClass(expectedClass));
    });

    QUnit.test('adds mouse listeners to the top level div', function(assert) {
        const sidebarComponent = renderSidebar();
        assert.equal(sidebarComponent.prop('onMouseEnter'), DEFAULT_PROPS.onMouseEnter);
        assert.equal(sidebarComponent.prop('onMouseLeave'), DEFAULT_PROPS.onMouseLeave);
    });

    QUnit.test('has correct sidebar header', function(assert) {
        const sidebarComponent = renderSidebar();
        const headerComponent = sidebarComponent.childAt(0);

        assert.equal(headerComponent.type(), CollectionSidebarHeader);
        assert.equal(headerComponent.prop('title'), DEFAULT_PROPS.collection.title);
        assert.equal(headerComponent.prop('ownerURL'), `https://twitch.tv/${DEFAULT_PROPS.collection.owner.name}`);
        assert.equal(headerComponent.prop('ownerName'), DEFAULT_PROPS.collection.owner.displayName);
        assert.equal(headerComponent.prop('itemsLength'), DEFAULT_PROPS.collection.items.length);
        assert.equal(headerComponent.prop('totalDuration'), DEFAULT_PROPS.collection.totalDuration);
        assert.equal(headerComponent.prop('onClose'), DEFAULT_PROPS.onClose);
    });

    QUnit.test('creates correct number of sidebar items with isActive, index, and key set correctly', function(assert) {
        const sidebarComponent = renderSidebar();
        const itemsContainerComponent = sidebarComponent.childAt(1);
        const itemComponents = itemsContainerComponent.children();

        assert.ok(itemsContainerComponent.hasClass('collection-sidebar-item-container'));
        assert.equal(itemComponents.length, DEFAULT_PROPS.collection.items.length);
        itemComponents.forEach((itemComponent, index) => {
            const isActive = index === DEFAULT_PROPS.currentItemIndex;
            // eslint-disable-next-line max-len
            const expectedKey = `${isActive}_${DEFAULT_PROPS.collection.items[index].itemId}_${DEFAULT_PROPS.collection.items[index].title}`;
            assert.equal(itemComponent.type(), CollectionSidebarItem);
            assert.equal(itemComponent.key(), expectedKey);
            assert.equal(itemComponent.prop('isActive'), isActive);
            assert.equal(itemComponent.prop('index'), index);
        });
    });

    QUnit.test('animated sidebar component correctly wraps collection sidebar component', function(assert) {
        assert.equal(AnimatedCollectionSidebar.displayName, 'WrappedAnimatedComponent(CollectionSidebarComponent)');
    });
});
