import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { CollectionTopBarComponent } from 'ui/components/collections/collection-topbar';
import { mockTranslateFunc } from '../../../../../utils/translate-hoc-helpers';

const DEFAULT_ARGS = {
    collectionItems: [],
    showDivider: false,
    streamId: '',
    title: '',
    t: mockTranslateFunc,
};

function renderCollectionTopBar(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <CollectionTopBarComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | collections | collection-topbar', function() {
    QUnit.test('renders the collection top bar with the correct title', function(assert) {
        const title = QUnit.config.current.testId;
        const component = renderCollectionTopBar({ title });

        assert.equal(
            component.find('.qa-collection-collapsed-title').text(),
            title
        );
    });

    QUnit.test('renders the collection top bar with the correct progress text', function(assert) {
        const component = renderCollectionTopBar({
            collectionItems: [{
                itemId: '1234',
            }, {
                itemId: '4567',
            }],
            streamId: '1234',
        });

        assert.equal(
            component.find('.qa-collection-collapsed-progress').text(),
            '1/2 videos'
        );
    });

    QUnit.test('shows the divider if showDivider is true', function(assert) {
        const component = renderCollectionTopBar();
        assert.equal(component.find('.collection-divider').getElements().length, 0);
        component.setProps({ showDivider: true });
        assert.equal(component.find('.collection-divider').getElements().length, 1);
    });

    QUnit.test('when clicked, showSidebar is called', function(assert) {
        const showSidebar = sinon.spy();
        const component = renderCollectionTopBar({ showSidebar });

        component.find('.collection-collapsed__layout').simulate('click');
        assert.equal(showSidebar.callCount, 1);
    });

    QUnit.test('when mouse enters, fires onMouseEnter callback', function(assert) {
        const onMouseEnter = sinon.spy();
        const component = renderCollectionTopBar({ onMouseEnter });

        component.childAt(0).simulate('mouseenter');
        assert.equal(onMouseEnter.callCount, 1);
    });

    QUnit.test('when mouse leaves, fire onMouseLeave callback', function(assert) {
        const onMouseLeave = sinon.spy();
        const component = renderCollectionTopBar({ onMouseLeave });

        component.childAt(0).simulate('mouseleave');
        assert.equal(onMouseLeave.callCount, 1);
    });
});
