import React from 'react';
import PropTypes from 'prop-types';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { makeAnimatedComponent } from 'ui/components/common/make-animated-component';
import { waitFor } from 'tests/utils/waitFor';

function Title({ title }) {
    return <h1>{title}</h1>;
}

Title.propTypes = {
    title: PropTypes.string,
};

const DEFAULT_OPTIONS = Object.freeze({
    showDuration: 0,
    hideDuration: 0,
    showClass: '',
    hideClass: '',
});

const DEFAULT_ARGS = {
    title: '',
    _next: () => {},
};

function renderAnimatedTitle(titleProps = {}, animationOpts = {}) {
    const options = assign({}, DEFAULT_OPTIONS, animationOpts);
    const AnimatedTitle = makeAnimatedComponent(Title, options);
    const args = assign({}, DEFAULT_ARGS, titleProps);
    const component = <AnimatedTitle {...args} />;
    return shallow(component);
}

reactTest('ui | components | common | make-animated-component', function() {
    QUnit.test('returns the existing component', function(assert) {
        const AnimatedTitle = makeAnimatedComponent(Title, DEFAULT_OPTIONS);
        const component = shallow(<AnimatedTitle />);

        assert.equal(component.type(), Title);
    });

    // eslint-disable-next-line max-len
    QUnit.test('sets className on the wrapped component if it is defined on the AnimatedComponent props', function(assert) {
        const propClassName = 'a prop class name';
        const AnimatedTitle = makeAnimatedComponent(Title, {});
        const component = shallow(<AnimatedTitle className={propClassName} />);

        assert.ok(component.hasClass(propClassName));
    });

    QUnit.test('sets specified props on the wrapped component', function(assert) {
        const expectedTitle = 'hi, i am a prop yo';
        const component = renderAnimatedTitle({ title: expectedTitle }, {});

        assert.equal(component.prop('title'), expectedTitle);
    });

    function testAnimationOnLoad(methodName) {
        QUnit.test(`shows component on ${methodName}`, function(assert) {
            const spy = sinon.spy();
            const animatedShowClass = QUnit.config.current.testId;
            const propClassName = 'a classname prop';
            const AnimatedTitle = makeAnimatedComponent(Title, { showClass: animatedShowClass });
            const component = shallow(<AnimatedTitle className={propClassName} />);

            const instance = component.instance();

            instance[methodName](spy);

            return waitFor(() => spy.calledOnce).
                then(() => {
                    component.update();
                    assert.equal(component.state().animationClass, animatedShowClass);
                    assert.ok(component.hasClass(animatedShowClass));
                    assert.ok(component.hasClass(propClassName));
                });
        });
    }

    testAnimationOnLoad('componentWillAppear');
    testAnimationOnLoad('componentWillEnter');

    QUnit.test('hides component on componentWillLeave', function(assert) {
        const spy = sinon.spy();
        const animatedHideClass = QUnit.config.current.testId;
        const propClassName = 'a classname prop';
        const AnimatedTitle = makeAnimatedComponent(Title, { hideClass: animatedHideClass });
        const component = shallow(<AnimatedTitle className={propClassName} />);

        const instance = component.instance();

        instance.componentWillLeave(spy);

        return waitFor(() => spy.calledOnce).
            then(() => {
                component.update();
                assert.equal(component.state().animationClass, animatedHideClass);
                assert.ok(component.hasClass(animatedHideClass));
                assert.ok(component.hasClass(propClassName));
            });
    });

    QUnit.test('if onAnimationEnd is specified, invoke on componentDidLeave', function(assert) {
        const onAnimationEnd = sinon.spy();
        const component = renderAnimatedTitle({
            onAnimationEnd,
        });
        const instance = component.instance();
        instance.componentDidLeave();
        assert.equal(onAnimationEnd.callCount, 1);
    });

    QUnit.test('if _transitionToNext is specified, invoke on componentDidLeave', function(assert) {
        const _transitionToNextComponent = sinon.spy();
        const component = renderAnimatedTitle({
            _transitionToNextComponent,
        });
        const instance = component.instance();
        instance.componentDidLeave();
        assert.equal(_transitionToNextComponent.callCount, 1);
    });
});
