import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { SelectDropdownComponent } from 'ui/components/common/select-dropdown';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';
import range from 'lodash/range';

const DEFAULT_ARGS = Object.freeze({
    className: '',
    currentValue: '',
    id: 'uniqueness',
    options: [],
    onSelect() {},
    t: mockTranslateFunc,
});

function renderSelectDropdown(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <SelectDropdownComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | common | select-dropdown', function() {
    QUnit.test('renders a select tag', function(assert) {
        const component = renderSelectDropdown();
        assert.equal(component.type(), 'select');
    });

    QUnit.test('renders select tag with className', function(assert) {
        const className = QUnit.config.current.testId;
        const component = renderSelectDropdown({ className });
        assert.ok(component.hasClass(className), 'sets correct className');
    });

    QUnit.test('sets the select value to currentValue', function(assert) {
        const currentValue = QUnit.config.current.testId;
        const component = renderSelectDropdown({ currentValue });

        assert.equal(component.props().value, currentValue, 'passes currentValue to select tag');
    });

    QUnit.test('creates a dropdown list from options', function(assert) {
        const options = range(3).map(value => {
            return {
                text: QUnit.config.current.testId,
                value,
            };
        });
        const component = renderSelectDropdown({ options });
        assert.ok(component.containsAllMatchingElements([
            <option
                key={0}
                value={0}
            >
                {QUnit.config.current.testId}
            </option>,
            <option
                key={1}
                value={1}
            >
                {QUnit.config.current.testId}
            </option>,
            <option
                key={2}
                value={2}
            >
                {QUnit.config.current.testId}
            </option>,
        ]), 'creates correct options');
    });

    QUnit.test('passes option value to onSelect on change and on blur', function(assert) {
        const onSelect = sinon.spy();
        const value = QUnit.config.current.testId;
        const component = renderSelectDropdown({ onSelect });

        component.simulate('change', {
            target: {
                value,
            },
        });

        assert.equal(onSelect.callCount, 1, 'called once');
        assert.equal(onSelect.firstCall.args[0], QUnit.config.current.testId, 'passes option value');
    });
});
