import React from 'react';
import TransitionGroup from 'react-transition-group/TransitionGroup';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { TransitionManager } from 'ui/components/common/transition-manager';
import { makeAnimatedComponent, DEFAULT_OPTIONS } from 'ui/components/common/make-animated-component';

// eslint-disable-next-line react/prop-types
function Title({ title }) {
    return <h1>{title}</h1>;
}

const AnimatedTitle = makeAnimatedComponent(Title, DEFAULT_OPTIONS);

const DEFAULT_ARGS = Object.freeze({
    className: '',
    children: <AnimatedTitle title={'default'} />,
    animate: true,
});

function renderTransitionManager(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <TransitionManager {...args} />;
    return shallow(component);
}

reactTest('ui | components | common | transition-mananger', function() {
    QUnit.test('wraps component in div if `animate` is false', function(assert) {
        const className = QUnit.config.current.testId;
        const component = renderTransitionManager({
            className,
            animate: false,
        });

        assert.ok(component.containsMatchingElement(
            <div className={className}>
                <AnimatedTitle title={'default'} />
            </div>
        ));
    });

    QUnit.test('wraps component in TransitionGroup if `animate` is true', function(assert) {
        const className = QUnit.config.current.testId;
        const component = renderTransitionManager({
            className,
        });

        assert.equal(component.type(), TransitionGroup);
    });

    QUnit.test('transitioning state is false on initial render', function(assert) {
        const component = renderTransitionManager();
        const { transitioning } = component.state();
        assert.equal(transitioning, false);
    });

    QUnit.test('if animated is true, it should pass `this.renderNextComponent` to its child', function(assert) {
        const component = renderTransitionManager();
        const instance = component.instance();

        const animatedTitle = component.childAt(0);
        const { _transitionToNextComponent } = animatedTitle.props();

        assert.equal(typeof _transitionToNextComponent, 'function', '_transitionToNextComponent is a function');
        assert.equal(instance.renderNextComponent, _transitionToNextComponent);
    });

    QUnit.test('passing a new child will set transitioning to true', function(assert) {
        const component = renderTransitionManager();
        const instance = component.instance();
        sinon.spy(instance, 'setState');

        const newAnimatedTitle = <AnimatedTitle title={'new child'} />;
        component.setProps({ children: newAnimatedTitle });

        assert.deepEqual(instance.setState.firstCall.args[0], { transitioning: true });
    });

    QUnit.test('if transitioning state is true, component will render a null child', function(assert) {
        const component = renderTransitionManager();
        component.setState({ transitioning: true });

        assert.ok(component.containsMatchingElement(
            <TransitionGroup component={'div'} />
        ));
    });

    // eslint-disable-next-line max-len
    QUnit.test('calling _transitionToNextComponent on the animated component will set transitioning state to false', function(assert) {
        const component = renderTransitionManager();
        const animatedTitle = component.childAt(0);
        const { _transitionToNextComponent } = animatedTitle.props();
        const instance = component.instance();

        sinon.spy(instance, 'setState');

        _transitionToNextComponent();

        assert.deepEqual(instance.setState.firstCall.args[0], { transitioning: false });
    });

    QUnit.test('if transitioning state is false, component will render its child', function(assert) {
        const component = renderTransitionManager();
        component.setState({ transitioning: false });

        assert.ok(component.containsMatchingElement(
            <TransitionGroup
                component={'div'}
            >
                <AnimatedTitle title={'default'} />
            </TransitionGroup>
        ));
    });
});
