import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import { ExtensionsConfirmationComponent } from 'ui/components/extensions-confirmation';
import sinon from 'sinon';
import { PRODUCT_PAYLOAD } from '../containers/extensions-confirmation.test';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    confirm() {},
    cancel() {},
    isLinked: true,
    isLoggedIn: true,
    productPayload: PRODUCT_PAYLOAD,
    t: mockTranslateFunc,
});

function render(overrides = {}, method = shallow) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ExtensionsConfirmationComponent {...args} />;
    return method(component);
}

reactTest('ui | components | extension-confirmation', function() {
    QUnit.test('renders buy and cancel', function(assert) {
        const component = render();
        assert.equal(component.find('[data-test-extension-confirmation-purchase]').length, 1, 'purchase present');
        assert.equal(component.find('[data-test-extension-confirmation-cancel]').length, 1, 'cancel present');
    });

    QUnit.test('buy is clickable and calls confirm', function(assert) {
        const confirm = sinon.spy();
        const component = render({
            confirm,
        });

        assert.equal(confirm.callCount, 0, 'precond - no confirm without click');
        component.find('[data-test-extension-confirmation-purchase]').simulate('click');

        assert.equal(confirm.callCount, 1, 'buying triggers confirm');
    });

    QUnit.test('cancel is clickable and clears UI', function(assert) {
        const cancel = sinon.spy();
        const component = render({
            cancel,
        });

        assert.equal(cancel.callCount, 0, 'precond - no cancel without click');
        component.find('[data-test-extension-confirmation-cancel]').simulate('click');

        assert.equal(cancel.callCount, 1, 'cancels upon click');
    });

    QUnit.test('logged out notice appears when isLoggedIn === false', function(assert) {
        const component = render({
            isLoggedIn: false,
        });

        assert.equal(
            component.find('[data-test-extensions-confirmation-logged-out]').length,
            1,
            'logged out UI is present'
        );
    });

    QUnit.test('link notice appears when isLinked === false', function(assert) {
        const component = render({
            isLinked: false,
        });

        assert.equal(
            component.find('[data-test-extensions-confirmation-link-notice]').length,
            1,
            'link notice UI is present'
        );
    });
});
