import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { Interpolate } from 'react-i18next';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { createExtensionParsed } from 'tests/fixtures/extensions';
import { EXTENSION_MENU_DETAILS } from 'util/extensions';
import {
    ExtensionAccessMenuComponent,
    MENU_TOGGLE_IDENTITY_BUTTON,
    MENU_TOGGLE_IDENTITY_LABEL,
    MENU_BACK_BUTTON,
    MENU_BACK_LABEL,
    MENU_SUMMARY_LABEL,
    GRANT_ACCESS_HEADER,
    REVOKE_ACCESS_HEADER,
    GRANT_ACCESS_SUMMARY,
    REVOKE_ACCESS_SUMMARY,
    MANAGE_PERMISSIONS_TEXT,
    MENU_MANAGE_PERMISSION_LINK,
} from 'ui/components/extensions/extension-access-menu';

const DEFAULT_ARGS = Object.freeze({
    extension: createExtensionParsed()[0],
    onMenuTransition() {},
    onIdentityToggle() {},
    trackEvent() {},
    t: string => string,
});

function renderMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ExtensionAccessMenuComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | extensions | extension-access-menu', function() {
    QUnit.test('when back button is clicked', function(assert) {
        const menuTransitionSpy = sinon.spy();
        const component = renderMenu({
            onMenuTransition: menuTransitionSpy,
        });

        const backButton = component.find(`.${MENU_BACK_BUTTON}`);
        backButton.simulate('click');
        assert.equal(menuTransitionSpy.callCount, 1,
            'menu transition is called');
        assert.equal(menuTransitionSpy.firstCall.args[0], EXTENSION_MENU_DETAILS,
            'menu transition is called with expected value');
    });

    QUnit.test('when toggle access button is clicked', function(assert) {
        const extension = createExtensionParsed()[0];
        const identityToggleSpy = sinon.spy();
        const trackEventSpy = sinon.spy();
        const component = renderMenu({
            extension,
            onIdentityToggle: identityToggleSpy,
            trackEvent: trackEventSpy,
        });

        const toggleIdentityButton = component.find(`.${MENU_TOGGLE_IDENTITY_BUTTON}`);
        toggleIdentityButton.simulate('click');
        assert.equal(identityToggleSpy.callCount, 1,
            'menu transition is called');
        assert.deepEqual(identityToggleSpy.firstCall.args[0], extension,
            'menu transition is called with the current extension');
        assert.equal(trackEventSpy.callCount, 1,
            'trackEvent is called');
        assert.deepEqual(trackEventSpy.firstCall.args, [
            'extension_ui_interaction_client',
            extension, {
                // eslint-disable-next-line camelcase
                extension_interaction: 'grant',
            },
        ], 'trackEvent is called with the expected arguments');
    });

    QUnit.test('when an extension without identity sharing granted is provided', function(assert) {
        const extension = createExtensionParsed()[0];
        extension.lastUserIdentityLinkState = false;
        const component = renderMenu({
            extension,
        });

        const expectedSummaryComponent = (
            <Interpolate
                i18nKey={GRANT_ACCESS_SUMMARY}
                extensionName={<b>{extension.name}</b>}
            />
        );

        const backButton = component.find(`.${MENU_BACK_LABEL}`);
        assert.equal(backButton.length, 1, 'back button label exists');
        assert.equal(backButton.text(), GRANT_ACCESS_HEADER, 'back button label has the correct text');

        const summaryLabel = component.find(`.${MENU_SUMMARY_LABEL}`);
        assert.equal(summaryLabel.length, 1, 'summary label exists');
        assert.ok(summaryLabel.containsMatchingElement(expectedSummaryComponent), 'summary label has the correct text');

        const toggleButton = component.find(`.${MENU_TOGGLE_IDENTITY_LABEL}`);
        assert.equal(toggleButton.length, 1, 'toggle button label exists');
        assert.equal(toggleButton.text(), GRANT_ACCESS_HEADER, 'toggle button label has the correct text');

        const managePermissionLink = component.find(`.${MENU_MANAGE_PERMISSION_LINK}`);
        assert.equal(managePermissionLink.length, 1, 'manage permission link exists');
        assert.equal(
            managePermissionLink.text(),
            MANAGE_PERMISSIONS_TEXT,
            'manage permission text has the correct text'
        );
    });

    QUnit.test('when an extension with identity sharing granted is provided', function(assert) {
        const extension = createExtensionParsed()[0];
        extension.lastUserIdentityLinkState = true;
        const component = renderMenu({
            extension,
        });

        const expectedSummaryComponent = (
            <Interpolate
                i18nKey={REVOKE_ACCESS_SUMMARY}
                extensionName={<b>{extension.name}</b>}
            />
        );

        const backButton = component.find(`.${MENU_BACK_LABEL}`);
        assert.equal(backButton.length, 1, 'back button label exists');
        assert.equal(backButton.text(), REVOKE_ACCESS_HEADER, 'back button label has the correct text');

        const summaryLabel = component.find(`.${MENU_SUMMARY_LABEL}`);
        assert.equal(summaryLabel.length, 1, 'summary label exists');
        assert.ok(summaryLabel.containsMatchingElement(expectedSummaryComponent), 'summary label has the correct text');

        const toggleButton = component.find(`.${MENU_TOGGLE_IDENTITY_LABEL}`);
        assert.equal(toggleButton.length, 1, 'toggle button label exists');
        assert.equal(toggleButton.text(), REVOKE_ACCESS_HEADER, 'toggle button label has the correct text');
    });
});
