import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { createExtensionParsed } from 'tests/fixtures/extensions';
import {
    ExtensionDetailsMenuComponent,
    TWITCH_EXT_DETAILS_BASE,
    EXT_BACK_BUTTON,
    EXT_REPORT_BUTTON,
    EXT_DETAILS_BUTTON,
    EXT_MANAGE_ACCESS_BUTTON,
} from 'ui/components/extensions/extension-details-menu';
import {
    EXTENSION_MENU_MAIN,
    EXTENSION_MENU_MANAGE_ACCESS,
    EXTENSION_MENU_REPORT,
    EXTENSION_ROLE_BROADCASTER,
} from 'util/extensions';

const DEFAULT_ARGS = Object.freeze({
    extension: createExtensionParsed()[0],
    isLoggedIn: true,
    showBackButton: true,
    windowObj: {
        open() {},
    },
    onMenuTransition() {},
    t: str => str,
});

function renderMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ExtensionDetailsMenuComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | extensions | extension-details-menu', function() {
    QUnit.test('when back button is clicked', function(assert) {
        const menuTransitionSpy = sinon.spy();
        const component = renderMenu({
            onMenuTransition: menuTransitionSpy,
        });

        const backButton = component.find(`.${EXT_BACK_BUTTON}`);
        backButton.simulate('click');
        assert.equal(menuTransitionSpy.callCount, 1,
            'menu transition is called');
        assert.equal(menuTransitionSpy.firstCall.args[0], EXTENSION_MENU_MAIN,
            'menu transition is called with expected value');
    });

    QUnit.test('when details is clicked', function(assert) {
        const extension = createExtensionParsed()[0];
        const menuTransitionSpy = sinon.spy();
        const windowOpenSpy = sinon.spy();
        const component = renderMenu({
            extension,
            onMenuTransition: menuTransitionSpy,
            windowObj: {
                open: windowOpenSpy,
            },
        });

        const detailsButton = component.find(`.${EXT_DETAILS_BUTTON}`);
        detailsButton.simulate('click');
        assert.equal(windowOpenSpy.callCount, 1,
            'window.open is called');

        const detailsUrl = `${TWITCH_EXT_DETAILS_BASE}/${extension.id}-${extension.version}`;
        assert.equal(windowOpenSpy.firstCall.args[0], detailsUrl,
            'window.open is called with the correct URL');
        assert.equal(windowOpenSpy.firstCall.args[1], '_blank',
            'window.open targets a new window');
    });

    QUnit.test('when showBackButton is false', function(assert) {
        const component = renderMenu({
            showBackButton: false,
        });
        assert.equal(component.find(`.${EXT_BACK_BUTTON}`).length, 0,
            'back button is not rendered');
    });

    QUnit.test('when not logged in', function(assert) {
        const component = renderMenu({
            isLoggedIn: false,
        });
        assert.equal(component.find(`.${EXT_MANAGE_ACCESS_BUTTON}`).length, 0,
            'manage access button is not rendered');
    });

    QUnit.test('when an extension without identity linking is set', function(assert) {
        const extension = createExtensionParsed()[0];
        extension.supportsIdentityLinking = false;
        const component = renderMenu({
            extension,
        });

        const userPermissionsButton = component.find(`.${EXT_MANAGE_ACCESS_BUTTON}`);
        assert.equal(userPermissionsButton.length, 0, 'manage access button element does not exist');
    });

    QUnit.test('when extension token role is broadcaster, user permission button does not appear', function(assert) {
        const extension = createExtensionParsed()[0];
        extension.token.role = EXTENSION_ROLE_BROADCASTER;
        const component = renderMenu({
            extension,
        });

        const userPermissionsButton = component.find(`.${EXT_MANAGE_ACCESS_BUTTON}`);
        assert.equal(userPermissionsButton.length, 0, 'user permissions button element does not exist');
    });

    QUnit.test('when manage access button is clicked', function(assert) {
        const menuTransitionSpy = sinon.spy();
        const component = renderMenu({
            onMenuTransition: menuTransitionSpy,
        });

        const accessButton = component.find(`.${EXT_MANAGE_ACCESS_BUTTON}`);
        accessButton.simulate('click');
        assert.equal(menuTransitionSpy.callCount, 1,
            'menu transition is called');
        assert.equal(menuTransitionSpy.firstCall.args[0], EXTENSION_MENU_MANAGE_ACCESS,
            'menu transition is called with expected value');
    });

    QUnit.test('when report button is clicked', function(assert) {
        const extension = createExtensionParsed()[0];
        const menuTransitionSpy = sinon.spy();
        const trackEventSpy = sinon.spy();
        const component = renderMenu({
            extension,
            onMenuTransition: menuTransitionSpy,
            trackEvent: trackEventSpy,
        });

        const reportButton = component.find(`.${EXT_REPORT_BUTTON}`);
        reportButton.simulate('click');
        assert.equal(menuTransitionSpy.callCount, 1,
            'menu transition is called');
        assert.equal(menuTransitionSpy.firstCall.args[0], EXTENSION_MENU_REPORT,
            'menu transition is called with expected value');
        assert.equal(trackEventSpy.callCount, 1,
            'trackEvent is called');
        assert.deepEqual(trackEventSpy.firstCall.args, [
            'extension_ui_interaction_client',
            extension, {
                // eslint-disable-next-line camelcase
                extension_interaction: 'report',
            },
        ], 'trackEvent is called with the expected arguments');
    });
});
