import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import assign from 'lodash/assign';
import sinon from 'sinon';
import classNames from 'classnames';
import { shallow } from 'enzyme';
import {
    ExtensionFollowModalComponent,
    MODAL_TITLE_HEADER,
    MODAL_TITLE_CHANNEL,
    MODAL_BUTTON_CONFIRM,
    MODAL_BUTTON_CANCEL,
    MODAL_NOTIFICATION_TOGGLE,
} from 'ui/components/extensions/extension-follow-modal';
import { createExtensionParsed } from 'tests/fixtures/extensions';
import { I18N_INSTANCE } from 'lang/i18n';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';
import { Interpolate } from 'react-i18next';

const DEFAULT_ARGS = {
    extension: createExtensionParsed(),
    options: {
        channel: 'twitch',
        isFollowing: false,
        isLoggedIn: true,
    },
    onClose() {},
    t: mockTranslateFunc,
    i18n: I18N_INSTANCE,
};

function renderFollowModal(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ExtensionFollowModalComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | extensions | extension-follow-modal', function() {
    QUnit.test('renders the a dialog with a prompt to follow if the isFollowing option is false', function(assert) {
        const component = renderFollowModal();
        const headerTitle = component.find(`.${MODAL_TITLE_HEADER}`);
        const confirmButton = component.find(`.${MODAL_BUTTON_CONFIRM}`);
        const cancelButton = component.find(`.${MODAL_BUTTON_CANCEL}`);
        const notificationToggle = component.find(`.${MODAL_NOTIFICATION_TOGGLE}`);

        const channelTitleComponent = (
            <span className={classNames(MODAL_TITLE_CHANNEL, 'pl-menu__item--label', 'extension-modal__title-channel')}>
                {'twitch'}
            </span>
        );

        const expectedHeaderComponent = (
            <Interpolate
                i18nKey={'You are about to follow {{channelName}}'}
                channelName={channelTitleComponent}
            />
        );

        assert.ok(
            headerTitle.containsMatchingElement(expectedHeaderComponent),
            'renders the proper title prompt'
        );

        assert.equal(confirmButton.length, 1, 'renders a confirm button');
        assert.equal(cancelButton.length, 1, 'renders a cancel button');
        assert.equal(notificationToggle.length, 1, 'renders a notification toggle');
    });

    QUnit.test('renders a dialog with a message to dismiss if the channel is null', function(assert) {
        const component = renderFollowModal({
            options: {
                channel: null,
                isFollowing: false,
                isLoggedIn: true,
            },
        });
        const headerTitle = component.find(`.${MODAL_TITLE_HEADER}`);
        const confirmButton = component.find(`.${MODAL_BUTTON_CONFIRM}`);
        const cancelButton = component.find(`.${MODAL_BUTTON_CANCEL}`);
        const notificationToggle = component.find(`.${MODAL_NOTIFICATION_TOGGLE}`);

        assert.equal(
            headerTitle.text(),
            'The channel you are attempting to follow does not exist',
            'renders the proper title prompt'
        );

        assert.equal(confirmButton.length, 0, 'does not render a confirm button');
        assert.equal(cancelButton.length, 1, 'renders a cancel button');
        assert.equal(notificationToggle.length, 0, 'does not render a notification toggle');
    });

    QUnit.test('renders a dialog with a message to dismiss if the isFollowing option is true', function(assert) {
        const component = renderFollowModal({
            options: {
                channel: 'twitch',
                isFollowing: true,
                isLoggedIn: true,
            },
        });
        const headerTitle = component.find(`.${MODAL_TITLE_HEADER}`);
        const confirmButton = component.find(`.${MODAL_BUTTON_CONFIRM}`);
        const cancelButton = component.find(`.${MODAL_BUTTON_CANCEL}`);
        const notificationToggle = component.find(`.${MODAL_NOTIFICATION_TOGGLE}`);

        const channelTitleComponent = (
            <span className={classNames(MODAL_TITLE_CHANNEL, 'pl-menu__item--label', 'extension-modal__title-channel')}>
                {'twitch'}
            </span>
        );

        const expectedHeaderComponent = (
            <Interpolate
                i18nKey={'You are already following {{channelName}}'}
                channelName={channelTitleComponent}
            />
        );

        assert.ok(
            headerTitle.containsMatchingElement(expectedHeaderComponent),
            'renders the proper title prompt'
        );

        assert.equal(confirmButton.length, 0, 'does not render a confirm button');
        assert.equal(cancelButton.length, 1, 'renders a cancel button');
        assert.equal(notificationToggle.length, 0, 'does not render a notification toggle');
    });

    QUnit.test('when the confirm button is clicked', function(assert) {
        const onCloseSpy = sinon.spy();
        const component = renderFollowModal({
            onClose: onCloseSpy,
        });

        const confirmButton = component.find(`.${MODAL_BUTTON_CONFIRM}`);
        confirmButton.simulate('click');
        assert.ok(
            onCloseSpy.calledWith({
                didFollow: true,
                notifications: false,
            }),
            'clicking the button calls onClose with didFollow true'
        );
    });

    QUnit.test('when the cancel button is clicked', function(assert) {
        const onCloseSpy = sinon.spy();
        const component = renderFollowModal({
            onClose: onCloseSpy,
        });

        const cancelButton = component.find(`.${MODAL_BUTTON_CANCEL}`);
        cancelButton.simulate('click');
        assert.ok(
            onCloseSpy.calledWith({
                didFollow: false,
                notifications: false,
            }),
            'clicking the button calls onClose with didFollow false'
        );
    });

    QUnit.test('when the notifications setting is toggled on', function(assert) {
        const onCloseSpy = sinon.spy();
        const component = renderFollowModal({
            onClose: onCloseSpy,
        });

        const confirmButton = component.find(`.${MODAL_BUTTON_CONFIRM}`);

        // normally this would be called by way of clicking the toggle element, but this is a shallow render
        component.instance().onNotificationToggle();

        assert.equal(component.state('notifications'), true);

        confirmButton.simulate('click');
        assert.ok(
            onCloseSpy.calledWith({
                didFollow: true,
                notifications: true,
            }),
            'clicking the confirm button calls onClose with didFollow true'
        );
    });
});
