import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ExtensionsMenuManager } from 'ui/components/extensions/extensions-menu-manager';
import { ExtensionListMenu } from 'ui/components/extensions/extension-list-menu';
import { ExtensionDetailsMenu } from 'ui/components/extensions/extension-details-menu';
import { ExtensionAccessMenu } from 'ui/components/extensions/extension-access-menu';
import { ExtensionReportMenu } from 'ui/components/extensions/extension-report-menu';
import { SETTINGS_INNER_PADDING } from 'ui/components/settings/menu-manager';
import {
    EXTENSION_MENU_MAIN,
    EXTENSION_MENU_DETAILS,
    EXTENSION_MENU_MANAGE_ACCESS,
    EXTENSION_MENU_REPORT,
} from 'util/extensions';
import { createExtensionParsed } from 'tests/fixtures/extensions';
import { buildFakeWindow } from 'tests/fakes/window.fake';

const DEFAULT_ARGS = {
    playerHeight: 100,
    isLoggedIn: false,
    windowObj: buildFakeWindow(),
    extensions: [],
    onIdentityToggle() {},
    trackEvent() {},
    handleMenuTransition() {},
};

function renderMenuManager(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ExtensionsMenuManager {...args} />;
    return shallow(component);
}

reactTest('ui | components | extensions | extensions-menu-manager', function() {
    QUnit.test('renders the extension list menu by default if there are no extensions', function(assert) {
        const component = renderMenuManager({
            extensions: [],
        });
        assert.equal(component.childAt(0).type(), ExtensionListMenu, 'renders main menu');
    });

    QUnit.test('renders the extension detail menu by default if there is one extension', function(assert) {
        const handleMenuTransitionSpy = sinon.spy();
        renderMenuManager({
            extensions: createExtensionParsed(),
            handleMenuTransition: handleMenuTransitionSpy,
        });

        assert.ok(handleMenuTransitionSpy.calledWith(EXTENSION_MENU_DETAILS), 'transitions to details');
    });

    QUnit.test('div has the max height set to playerHeight - inner padding', function(assert) {
        const playerHeight = 500;
        const component = renderMenuManager({
            playerHeight,
        });

        assert.equal(component.props().style.maxHeight, `${playerHeight - SETTINGS_INNER_PADDING}px`);
    });

    QUnit.module('when currentMenu is extensions list menu', function(hooks) {
        hooks.beforeEach(function() {
            this.component = renderMenuManager({
                submenu: EXTENSION_MENU_MAIN,
            });
        });

        QUnit.test('renders the ExtensionListMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), ExtensionListMenu);
        });
    });

    QUnit.module('when currentMenu is extension details menu', function(hooks) {
        let handleMenuTransitionSpy;

        hooks.beforeEach(function() {
            handleMenuTransitionSpy = sinon.spy();

            this.component = renderMenuManager({
                submenu: EXTENSION_MENU_DETAILS,
                handleMenuTransition: handleMenuTransitionSpy,
            });
            this.component.setState({
                currentExtension: DEFAULT_ARGS.extensions[0],
            });
        });

        QUnit.test('renders the ExtensionDetailsMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), ExtensionDetailsMenu);
        });

        QUnit.test('it passes handleMenuTransition to the ExtensionDetailsMenu', function(assert) {
            this.component.find(ExtensionDetailsMenu).props().onMenuTransition(EXTENSION_MENU_DETAILS);
            assert.ok(handleMenuTransitionSpy.calledWith(EXTENSION_MENU_DETAILS));
        });
    });

    QUnit.module('when currentMenu is extension manage access', function(hooks) {
        let handleMenuTransitionSpy;

        hooks.beforeEach(function() {
            handleMenuTransitionSpy = sinon.spy();

            this.component = renderMenuManager({
                submenu: EXTENSION_MENU_MANAGE_ACCESS,
                handleMenuTransition: handleMenuTransitionSpy,
            });
            this.component.setState({
                currentExtension: DEFAULT_ARGS.extensions[0],
            });
        });

        QUnit.test('render ExtensionAccessMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), ExtensionAccessMenu);
        });

        QUnit.test('it passes handleMenuTransition to the ExtensionAccessMenu', function(assert) {
            this.component.find(ExtensionAccessMenu).props().onMenuTransition(EXTENSION_MENU_DETAILS);
            assert.ok(handleMenuTransitionSpy.calledWith(EXTENSION_MENU_DETAILS));
        });
    });

    QUnit.module('when currentMenu is extension report', function(hooks) {
        let handleMenuTransitionSpy;

        hooks.beforeEach(function() {
            handleMenuTransitionSpy = sinon.spy();

            this.component = renderMenuManager({
                submenu: EXTENSION_MENU_REPORT,
                handleMenuTransition: handleMenuTransitionSpy,
            });
            this.component.setState({
                currentExtension: DEFAULT_ARGS.extensions[0],
            });
        });

        QUnit.test('render ExtensionReportMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), ExtensionReportMenu);
        });

        QUnit.test('it passes handleMenuTransition to the ExtensionReportMenu', function(assert) {
            this.component.find(ExtensionReportMenu).props().onMenuTransition(EXTENSION_MENU_DETAILS);
            assert.ok(handleMenuTransitionSpy.calledWith(EXTENSION_MENU_DETAILS));
        });
    });

    QUnit.test('trackEvent is called on mount when there is a single extension', function(assert) {
        const extensions = createExtensionParsed();
        const trackEventSpy = sinon.spy();
        renderMenuManager({
            extensions,
            trackEvent: trackEventSpy,
        });

        assert.equal(trackEventSpy.callCount, 1,
            'trackEvent was called once');
        assert.deepEqual(trackEventSpy.firstCall.args, [
            'extension_setting_click',
            extensions[0],
        ], 'trackEvent was called with the expected arguments');
    });

    QUnit.test('trackEvent is called on extension click', function(assert) {
        const trackEventSpy = sinon.spy();
        const component = renderMenuManager({
            extensions: [],
            trackEvent: trackEventSpy,
        });
        const instance = component.instance();
        const newExtension = createExtensionParsed();

        instance.handleExtensionClick(newExtension);

        assert.equal(trackEventSpy.callCount, 1,
            'trackEvent was called once');
        assert.deepEqual(trackEventSpy.firstCall.args, [
            'extension_setting_click',
            newExtension,
        ], 'trackEvent was called with the expected arguments');
    });
});
