import React from 'react';
import assign from 'lodash/assign';
import pick from 'lodash/pick';
import sinon from 'sinon';
import { stringify } from 'query-string';
import { reactUnitTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { createExtensionParsed } from 'tests/fixtures/extensions';
import {
    ExtensionReportMenuComponent,
    MENU_BACK_BUTTON,
    EXT_SUBMIT_BUTTON,
    EXT_SUBMIT_LABEL,
} from 'ui/components/extensions/extension-report-menu';
import {
    EXTENSION_MENU_DETAILS,
} from 'util/extensions';
import { I18N_INSTANCE } from 'lang/i18n';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    channel: 'twitch',
    authenticityToken: '0',
    extension: createExtensionParsed()[0],
    onMenuTransition() {},
    t: mockTranslateFunc,
    i18n: I18N_INSTANCE,
});

function renderMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ExtensionReportMenuComponent {...args} />;
    return shallow(component);
}

reactUnitTest('ui | components | extensions | extension-report-menu', function(hooks) {
    hooks.beforeEach(function() {
        this.api.expectPostExtensionReport();
    });

    QUnit.test('when back button is clicked', function(assert) {
        const menuTransitionSpy = sinon.spy();
        const component = renderMenu({
            onMenuTransition: menuTransitionSpy,
        });

        const backButton = component.find(`.${MENU_BACK_BUTTON}`);
        backButton.simulate('click');
        assert.equal(menuTransitionSpy.callCount, 1,
            'menu transition is called');
        assert.equal(menuTransitionSpy.firstCall.args[0], EXTENSION_MENU_DETAILS,
            'menu transition is called with expected value');
    });

    QUnit.test('extension report form rendered as expected', function(assert) {
        const component = renderMenu();
        const selectElement = component.find('select');
        assert.ok(selectElement.length, 1, 'form contains a select element');
        assert.equal(selectElement.prop('name'), 'reason', 'form select element has correct name');

        const textareaElement = component.find('textarea');
        assert.ok(textareaElement.length, 1, 'form contains a textarea element');
        assert.equal(textareaElement.prop('name'), 'description', 'form select element has correct name');

        const submitButton = component.find(`.${EXT_SUBMIT_BUTTON}`);
        assert.ok(submitButton.length, 1, 'form contains a submit button element');
        assert.equal(submitButton.prop('onClick'), component.instance().handleSubmit);
    });

    QUnit.test('send button disabled by default', function(assert) {
        const component = renderMenu();
        const submitButton = component.find(`.${EXT_SUBMIT_BUTTON}`);
        assert.equal(submitButton.prop('disabled'), true, 'button is disabled by default');
    });

    QUnit.test('can\'t submit form with no input', function(assert) {
        const component = renderMenu();
        sinon.spy(component.instance(), 'handleSubmit');
        const submitButton = component.find(`.${EXT_SUBMIT_BUTTON}`);
        submitButton.simulate('click');
        assert.equal(component.instance().handleSubmit.callCount, 0, 'onSubmit not fired');
    });

    QUnit.test('can submit form with description and reason', function(assert) {
        const component = renderMenu();
        sinon.spy(component.instance(), 'handleSubmit');

        component.setState({
            description: 'test - just resolve',
            reason: 'other',
        });
        component.update();

        let submitButton = component.find(`.${EXT_SUBMIT_BUTTON}`);
        assert.equal(submitButton.prop('disabled'), false, 'button enabled after inputs are entered');
        submitButton.simulate('click');
        assert.equal(component.instance().handleSubmit.callCount, 1, 'handleSubmit called');

        // normally handleSubmit() will set this value but since our handleSubmit is a spy, we set it in the test
        component.setState({
            sent: true,
        });
        component.update();
        submitButton = component.find(`.${EXT_SUBMIT_BUTTON}`);
        const submitLabel = component.find(`.${EXT_SUBMIT_LABEL}`);
        assert.equal(submitLabel.text(), 'Sent!', 'send button text changes to sent!');
        assert.equal(submitButton.prop('disabled'), true, 'button disabled after sending');
    });

    QUnit.test('handleSubmit posts report correctly', function(assert) {
        const extension = createExtensionParsed()[0];
        const component = renderMenu({ extension });
        sinon.spy(component.instance(), 'handleSubmit');

        component.setState({
            description: 'test - just resolve',
            reason: 'other',
        });
        component.update();

        const expectedUrl = 'https://www.twitch.tv/user/report';
        return component.instance().handleSubmit().then(() => {
            const req = this.api.lastCall();
            component.update();

            // add our extension data to description for comparing requestBody
            const description = [
                component.state('description'),
                `Extension ID: ${extension.id}`,
                `Extension Name: ${extension.name}`,
                `Extension Version: ${extension.version}`,
            ].join('\n');
            const expectedFields = [
                'reported_user',
                'authenticity_token',
                'tos_ban',
                'reason',
                'description',
            ];

            const expectedBody = stringify(pick(assign({}, component.state(), { description }), expectedFields));
            assert.ok(this.api.called(expectedUrl), 'report request sent');
            assert.equal(req[0], expectedUrl, 'expected request to be sent to correct url');
            assert.equal(req[1].body, expectedBody, 'expected request bodies to match');
        });
    });
});
