import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { shallow } from 'enzyme';
import {
    ExtensionsModalManager,
    MODAL_BACKDROP,
} from 'ui/components/extensions/extensions-modal-manager';
import { ExtensionFollowModal } from 'ui/components/extensions/extension-follow-modal';
import { ExtensionFeatureNeedsAuthModal } from 'ui/components/extensions/extension-feature-needs-auth-modal';
import { createExtensionParsed } from 'tests/fixtures/extensions';
import { FunctionAction } from 'extension-coordinator/lib/constants';

const DEFAULT_ARGS = {
    playerHeight: 500,
    extensions: createExtensionParsed(),
    modalRequest: null,
    onModalClose() {},
};

const FOLLOW_MODAL_REQUEST = {
    action: FunctionAction.FollowAction,
    options: {},
    resultCallback: sinon.spy(),
    defaultResult: {
        didFollow: false,
        notifications: false,
    },
};

const LOGIN_MODAL_REQUEST = {
    action: FunctionAction.LoginRequest,
};

function renderModalManager(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ExtensionsModalManager {...args} />;
    return shallow(component);
}

reactTest('ui | components | extensions | extensions-modal-manager', function(hooks) {
    hooks.afterEach(function() {
        FOLLOW_MODAL_REQUEST.resultCallback.reset();
    });

    QUnit.test('when there is no modal request', function(assert) {
        const component = renderModalManager();
        assert.equal(component.type(), null, 'renders nothing');
    });

    QUnit.test('when there is a follow modal request', function(assert) {
        const component = renderModalManager({
            modalRequest: FOLLOW_MODAL_REQUEST,
        });

        assert.equal(
            component.childAt(0).childAt(0).type(),
            ExtensionFollowModal,
            'renders the ExtensionFollowModal component'
        );
    });

    QUnit.test('when there is a login modal request', function(assert) {
        const component = renderModalManager({
            modalRequest: LOGIN_MODAL_REQUEST,
        });

        assert.equal(
            component.childAt(0).childAt(0).type(),
            ExtensionFeatureNeedsAuthModal,
            'renders the ExtensionFeatureNeedsAuthModal component'
        );
    });

    QUnit.test('when the modal backdrop is clicked', function(assert) {
        const modalCloseCallback = sinon.spy();

        const component = renderModalManager({
            modalRequest: FOLLOW_MODAL_REQUEST,
            onModalClose: modalCloseCallback,
        });

        const backdropElement = component.find(`.${MODAL_BACKDROP}`);

        backdropElement.simulate('click');
        assert.ok(modalCloseCallback.called, 'should call the onModalClose callback');
        assert.ok(
            FOLLOW_MODAL_REQUEST.resultCallback.calledWith(FOLLOW_MODAL_REQUEST.defaultResult),
            'should call the result callback with the default result'
        );
    });
});

