import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { NotificationPopUp, POPUP_CLASS, CLOSE_BUTTON_CLASS } from 'ui/components/notifications/notification-popup';

const DEFAULT_ARGS = Object.freeze({
    isOpen: true,
    onClose: sinon.spy(),
});

function renderPopUp(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <NotificationPopUp {...args} />;
    return shallow(component);
}

reactTest('ui | components | notification-popup', function() {
    QUnit.test('returns a div with expected classname', function(assert) {
        const component = renderPopUp();
        assert.equal(component.type(), 'div');
        assert.ok(component.hasClass(POPUP_CLASS));
    });

    QUnit.test('close button should have expected class', function(assert) {
        const component = renderPopUp();
        const closeButton = component.find('button');
        assert.ok(closeButton.hasClass(CLOSE_BUTTON_CLASS));
    });

    QUnit.test('when close button is clicked, onClose is called', function(assert) {
        const onClose = sinon.spy();
        const component = renderPopUp({
            onClose,
        });
        const closeButton = component.find('button');
        closeButton.simulate('click');
        assert.equal(onClose.callCount, 1);
    });

    QUnit.test('children of popup should live in content section', function(assert) {
        const popUpWithChildren = shallow(
            <NotificationPopUp isOpen={true}>
                <p> Child 1 </p>
                <p> Child 2 </p>
            </NotificationPopUp>
        );
        assert.equal(popUpWithChildren.find('p').length, 2, 'finds two p tags');
        assert.ok(popUpWithChildren.contains(<p> Child 1 </p>));
        assert.ok(popUpWithChildren.contains(<p> Child 2 </p>));
    });
});
