import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import sinon from 'sinon';
import { AgeGateOverlayComponent } from 'ui/components/overlays/age-gate-overlay';
import { sessionStore } from 'tests/fakes/local-store.fake';

const DEFAULT_ARGS = Object.freeze({
    handleAgeGatePassed: () => { },
    requiredAge: 21,
    channelName: 'a channel name',
    t() {},
});

function renderAgeGateOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <AgeGateOverlayComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | overlays | age gate overlay', function() {
    QUnit.test('should render a div with the correct classes', function(assert) {
        const component = renderAgeGateOverlay();
        const expectedClassNames = [
            'player-overlay',
            'pl-age-restriction-overlay--wall',
            'pl-age-restriction-overlay',
            'js-age-restriction-overlay',
        ];
        assert.equal(component.type(), 'div', 'has correct type');
        assert.ok(expectedClassNames.every(
            componentClass => component.hasClass(componentClass)), 'has expected class names');
    });

    QUnit.test('should render an inner div with center content classes', function(assert) {
        const component = renderAgeGateOverlay();
        assert.equal(component.childAt(0).type(), 'div', 'has correct type');
        assert.ok(component.childAt(0).hasClass('player-center-content'), 'has player-center-content class');
    });

    QUnit.test('should render age gate class', function(assert) {
        const component = renderAgeGateOverlay();
        const { className } = component.find('.pl-age-gate').props();
        assert.equal(className, 'pl-age-gate js-age-gate', 'has expected class names');
    });

    reactTest('renders default date selector form', function() {
        QUnit.test('should render age gate form classes', function(assert) {
            const component = renderAgeGateOverlay();
            component.setState({
                ageGatesFailedCount: 0,
            });
            const { className } = component.find('.player-age-gate-form').props();
            assert.equal(className, 'player-age-gate-form js-age-gate', 'has expected class names');
        });

        QUnit.test('should render age gate notification classes', function(assert) {
            const component = renderAgeGateOverlay();
            component.setState({
                ageGatesFailedCount: 0,
            });
            const { className } = component.find('.player-age-gate-notificiation').props();
            assert.equal(className, 'player-age-gate-notificiation', 'has expected class names');
        });

        QUnit.test('should render age gate date picker classes', function(assert) {
            const component = renderAgeGateOverlay();
            component.setState({
                ageGatesFailedCount: 0,
            });
            const { className } = component.find('.player-datepicker').props();
            assert.equal(className, 'player-datepicker', 'has expected class names');
        });

        QUnit.test('should render date picker', function(assert) {
            const component = renderAgeGateOverlay();
            component.setState({
                ageGatesFailedCount: 0,
            });

            const datePicker = component.find('.player-datepicker');
            assert.equal(datePicker.length, 1, 'should have a date picker');

            [
                '.js-select-month',
                '.js-select-day',
                '.js-select-year',
            ].forEach(className => {
                assert.equal(datePicker.find(`${className}`).length, 1, `has ${className} class`);
            });
        });

        QUnit.test('should populate month drop down', function(assert) {
            const component = renderAgeGateOverlay();
            const instance = component.instance();
            sinon.spy(instance, '_populateMonthDropdown');
            component.setState({
                ageGatesFailedCount: 0,
                monthValue: '1',
            });

            assert.ok(instance._populateMonthDropdown.calledOnce, '_populateMonthDropdown called');
        });

        QUnit.test('should populate year drop down', function(assert) {
            const component = renderAgeGateOverlay();
            const instance = component.instance();
            sinon.spy(instance, '_populateYearDropdown');
            component.setState({
                ageGatesFailedCount: 0,
            });

            assert.ok(instance._populateYearDropdown.calledOnce, '_populateYearDropdown called');
        });

        QUnit.test('should populate day drop down', function(assert) {
            const component = renderAgeGateOverlay();
            const instance = component.instance();
            sinon.spy(instance, '_populateDayDropdown');
            component.setState({
                ageGatesFailedCount: 0,
            });

            assert.ok(instance._populateDayDropdown.calledOnce, '_populateDayDropdown called');
        });
        QUnit.test('should render age gate submit button', function(assert) {
            const component = renderAgeGateOverlay();
            component.setState({
                ageGatesFailedCount: 0,
            });

            assert.equal(component.find('.js-age-gate-submit').length, 1, 'has js-age-gate class');
        });

        QUnit.test('should render age gate warning icon', function(assert) {
            const component = renderAgeGateOverlay();
            component.setState({
                ageGatesFailedCount: 0,
            });

            assert.ok(component.find('.player-age-gate-warning').length, 1, 'has player-age-gate-warning class');
            assert.ok(component.find('.player-age-gate--icon').length, 1, 'has player-age-gate--icon class');
        });
    });

    reactTest('renders failed once overlay', function() {
        QUnit.test('should render age gate form classes', function(assert) {
            const component = renderAgeGateOverlay();
            component.setState({
                ageGatesFailedCount: 1,
            });

            assert.ok(component.find('.player-age-gate-failed-once').length, 1,
                'has player-age-gate-failed-once class');
            assert.ok(component.find('.player-age-gate--fail-icon').length, 1, 'has player-age-gate--fail-icon class');
            assert.ok(component.find('.age-gate-locked-out-label').length, 1, 'has age-gate-locked-out-label class');
        });
    });

    reactTest('renders locked out overlay', function() {
        QUnit.test('should render age gate form classes', function(assert) {
            const component = renderAgeGateOverlay();
            component.setState({
                ageGatesFailedCount: 2,
            });

            assert.ok(component.find('.pl-age-gate-locked-out').length, 1, 'has pl-age-gate-locked-out class');
            assert.ok(component.find('.player-age-gate--fail-icon').length, 1, 'has player-age-gate--fail-icon class');
            assert.ok(component.find('.age-gate-locked-out-label').length, 1, 'has age-gate-locked-out-label class');
        });
    });

    reactTest('on submitting age', function(hooks) {
        hooks.beforeEach(function() {
            sessionStore.clear();
        });

        function testAgeRequirement(requiredAge) {
            QUnit.test(`on ${requiredAge}+ channel, if over ${requiredAge}`, function(assert) {
                const handleAgeGatePassed = sinon.spy();
                const component = renderAgeGateOverlay({
                    handleAgeGatePassed,
                    requiredAge,
                });
                component.setState({
                    monthValue: '1',
                    dayValue: '1',
                    yearValue: ((new Date()).getFullYear() - requiredAge - 1).toString(),
                });
                const fakeEvent = {
                    preventDefault: sinon.spy(),
                };
                component.find('.js-age-gate-submit').simulate('click', fakeEvent);

                assert.equal(handleAgeGatePassed.callCount, 1, 'handleAgeGatePassed called');
            });

            QUnit.test(`on ${requiredAge}+ channel, if under ${requiredAge}`, function(assert) {
                const handleAgeGatePassed = sinon.spy();
                const component = renderAgeGateOverlay({
                    handleAgeGatePassed,
                    requiredAge,
                });
                component.setState({
                    monthValue: '1',
                    dayValue: '1',
                    yearValue: ((new Date()).getFullYear() - requiredAge + 1).toString(),
                });
                const fakeEvent = {
                    preventDefault: sinon.spy(),
                };
                component.find('.js-age-gate-submit').simulate('click', fakeEvent);

                assert.equal(handleAgeGatePassed.callCount, 0, 'handleAgeGatePassed is not called');
            });
        }

        testAgeRequirement(21);
        testAgeRequirement(20);
        testAgeRequirement(19);
        testAgeRequirement(18);
    });
});
