import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { UnmuteOverlayComponent, AnimatedUnmuteOverlay } from 'ui/components/overlays/unmute-overlay';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';
import assign from 'lodash/assign';

const DEFAULT_PROPS = Object.freeze({
    onClick: () => {},
    className: '',
    t: mockTranslateFunc,
});

function renderOverlay(propOverrides = {}) {
    const props = assign({}, DEFAULT_PROPS, propOverrides);
    const component = <UnmuteOverlayComponent {...props} />;
    return shallow(component);
}

reactTest('ui | components | overlays | unmute-overlay', function() {
    QUnit.test('has correct class on top level', function(assert) {
        const overlayComponent = renderOverlay();
        assert.ok(overlayComponent.hasClass('pl-overlay'), 'has overlay class');
        assert.ok(overlayComponent.hasClass('pl-overlay--unmute'), 'has unmute overlay class');
    });

    QUnit.test('adds class from className prop to top level div', function(assert) {
        const expectedClass = 'an expected class';
        const overlayComponent = renderOverlay({ className: expectedClass });
        assert.ok(overlayComponent.hasClass('pl-overlay'), 'has overlay class');
        assert.ok(overlayComponent.hasClass('pl-overlay--unmute'), 'has unmute overlay class');
        assert.ok(overlayComponent.hasClass(expectedClass), 'has prop className');
    });

    QUnit.test('adds on click and key up listeners to the top level div', function(assert) {
        const overlayComponent = renderOverlay();
        assert.equal(overlayComponent.prop('onClick'), DEFAULT_PROPS.onClick, 'has onclick handler');
        assert.equal(overlayComponent.prop('onKeyUp'), DEFAULT_PROPS.onClick, 'has onkeyup handler');
    });

    QUnit.test('has unmute icon', function(assert) {
        const overlayComponent = renderOverlay();
        const iconComponent = overlayComponent.childAt(0).childAt(0);

        assert.equal(iconComponent.type(), 'svg', 'is svg type');
        assert.ok(iconComponent.hasClass('pl-overlay__icon--unmute'), 'has icon class');
        assert.equal(iconComponent.childAt(0).prop('xlinkHref'), '#icon_volumemute', 'has xlinkhref');
    });

    QUnit.test('has translated string', function(assert) {
        const expectedString = 'Tap to unmute';
        const overlayComponent = renderOverlay();
        const textComponent = overlayComponent.childAt(0).childAt(1);

        assert.equal(textComponent.type(), 'p', 'is p type');
        assert.ok(textComponent.hasClass('pl-overlay__text--unmute'), 'has text class');
        assert.equal(textComponent.text(), expectedString, 'has correct translated string');
    });

    // eslint-disable-next-line max-len
    QUnit.test('animated unmute overlay component correctly wraps translated unmute overlay component', function(assert) {
        assert.equal(AnimatedUnmuteOverlay.displayName, 'WrappedAnimatedComponent(Translate(UnmuteOverlayComponent))');
    });
});
