import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow, mount } from 'enzyme';
import {
    AutoplayingPinnedVODComponent,
    AUTOPLAY_PERIOD_IN_MS,
} from 'ui/components/recommendations/autoplaying-pinned-vod';
import { PlayerCard } from 'ui/components/recommendations/cards/player-card';
import { Button } from 'ui/components/buttons/base-button';
import { NORMALIZED_V3_RECOMMENDED_VIDEOS } from 'tests/fixtures/recommendations';
import { getPercentage } from 'util/math';
import assign from 'lodash/assign';
import sinon from 'sinon';

const DEFAULT_PROPS = {
    video: NORMALIZED_V3_RECOMMENDED_VIDEOS[0],
    onSelect: () => {},
    windowObj: {
        clearInterval: () => {},
        setInterval: () => {},
        Date: {
            now: () => 0,
        },
    },
    onClose: () => {},
    width: 0,
    t: () => 'a string',
    trackEvent: () => {},
};

function shallowRenderPinnedVOD(propOverrides = {}) {
    const props = assign({}, DEFAULT_PROPS, propOverrides);
    const component = <AutoplayingPinnedVODComponent {...props} />;
    return shallow(component);
}

function mountPinnedVOD(propOverrides = {}) {
    const props = assign({}, DEFAULT_PROPS, propOverrides);
    const component = <AutoplayingPinnedVODComponent {...props} />;
    return mount(component);
}

reactTest('ui | components | recommendations | autoplaying-pinned-vod', function(hooks) {
    hooks.beforeEach(function() {
        this.intervalID = 123456;
        this.startDateNow = 100000;
        this.windowObj = {
            clearInterval: sinon.spy(),
            setInterval: sinon.stub().returns(this.intervalID),
            Date: {
                now: () => this.startDateNow,
            },
        };
    });

    QUnit.test('has correct class', function(assert) {
        const pinnedVODComponent = shallowRenderPinnedVOD();
        assert.ok(pinnedVODComponent.hasClass('pl-pinned-item'));
        assert.ok(pinnedVODComponent.hasClass('pl-pinned-item--centered'));
    });

    QUnit.test('has pinned item title as the first child', function(assert) {
        const pinnedVODComponent = shallowRenderPinnedVOD();
        const pinnedItemTitleComponent = pinnedVODComponent.childAt(0);

        assert.ok(pinnedItemTitleComponent.hasClass('pl-flex'));
        assert.ok(pinnedItemTitleComponent.hasClass('pl-pinned-item__title'));
    });

    QUnit.test('the pinned item title text is properly translated and highlighted', function(assert) {
        const translatedTitleText = 'a correctly translated string';
        const translatedTitleHighlight = 'another correctly translated string';
        const translateStub = sinon.stub();
        translateStub.withArgs('Playing in').returns(translatedTitleText);
        translateStub.withArgs('{{timeRemaining}} second', {
            timeRemaining: Math.round(AUTOPLAY_PERIOD_IN_MS / 1000),
            count: AUTOPLAY_PERIOD_IN_MS,
        }).returns(translatedTitleHighlight);

        const pinnedVODComponent = shallowRenderPinnedVOD({
            t: translateStub,
        });

        const pinnedItemTitleTextComponent = pinnedVODComponent.childAt(0).childAt(0);

        assert.ok(pinnedItemTitleTextComponent.hasClass('pl-flex__item'));
        assert.ok(pinnedItemTitleTextComponent.hasClass('pl-flex__item--grow'));

        assert.equal(pinnedItemTitleTextComponent.childAt(0).text(), translatedTitleText);
        assert.equal(pinnedItemTitleTextComponent.childAt(1).type(), 'span');
        assert.ok(pinnedItemTitleTextComponent.childAt(1).hasClass('pl-pinned-item__title-highlight'));
        assert.equal(pinnedItemTitleTextComponent.childAt(1).text(), ` ${translatedTitleHighlight}`);
    });

    QUnit.test('the pinned item title has a cancel button', function(assert) {
        const translatedButtonText = 'a correctly translated string';
        const translateStub = sinon.stub();
        translateStub.withArgs('Cancel').returns(translatedButtonText);
        const pinnedVODComponent = shallowRenderPinnedVOD({
            t: translateStub,
        });
        const pinnedItemTitleComponent = pinnedVODComponent.childAt(0);
        const cancelButtonComponent = pinnedItemTitleComponent.childAt(1);

        assert.equal(cancelButtonComponent.type(), Button);
        assert.equal(cancelButtonComponent.childAt(0).text(), translatedButtonText);
        assert.ok(cancelButtonComponent.hasClass('pl-rec__cancel'));
        assert.ok(cancelButtonComponent.hasClass('pl-button'));
        assert.ok(cancelButtonComponent.hasClass('pl-button--hollow'));
        assert.ok(cancelButtonComponent.hasClass('pl-button--white'));
    });

    QUnit.test('the cancel button onClick handler is clears the interval and calls onClose', function(assert) {
        const onCloseHandler = sinon.spy();
        const pinnedVODComponent = mountPinnedVOD({
            windowObj: this.windowObj,
            onClose: onCloseHandler,
        });

        const pinnedItemTitleComponent = pinnedVODComponent.childAt(0).childAt(0);
        const cancelButtonComponent = pinnedItemTitleComponent.childAt(1);
        const buttonOnClickHandler = cancelButtonComponent.prop('onClick');

        assert.equal(this.windowObj.setInterval.callCount, 1);
        assert.equal(this.windowObj.clearInterval.callCount, 0);
        assert.equal(onCloseHandler.callCount, 0);

        buttonOnClickHandler();

        assert.equal(this.windowObj.setInterval.callCount, 1);
        assert.equal(this.windowObj.clearInterval.callCount, 1);
        assert.equal(this.windowObj.clearInterval.firstCall.args[0], this.intervalID);
        assert.equal(onCloseHandler.callCount, 1);
        assert.ok(this.windowObj.clearInterval.calledBefore(onCloseHandler));
    });

    QUnit.test('has correctly populated player card as the second child', function(assert) {
        const pinnedVODComponent = shallowRenderPinnedVOD();
        const cardComponent = pinnedVODComponent.childAt(1);

        assert.ok(cardComponent.containsMatchingElement(
            <PlayerCard
                className="pl-rec__item"
                thumbnailURL={DEFAULT_PROPS.video.thumbnailURL}
                selectItem={DEFAULT_PROPS.video}
                progressBarPercentage={getPercentage(pinnedVODComponent.state().timeRemaining, AUTOPLAY_PERIOD_IN_MS)}
                hasThumbZoom={true}
                hasBorder={true}
            />
        ));
    });

    QUnit.test('for player card prop `info`, has VOD SVG and correctly translated string', function(assert) {
        const translatedString = 'a translated string';
        const translateStub = sinon.stub().withArgs('Most Recent Video').returns(translatedString);
        const pinnedVODComponent = shallowRenderPinnedVOD({
            t: translateStub,
        });

        const cardComponent = pinnedVODComponent.childAt(1);
        const cardInfoComponent = shallow(cardComponent.prop('info'));
        const VODSVGComponent = cardInfoComponent.childAt(0);
        const textComponent = cardInfoComponent.childAt(1);

        assert.equal(cardInfoComponent.type(), 'span');
        assert.ok(VODSVGComponent.containsMatchingElement(
            <figure className="pl-card__smallicon">
                <svg>
                    <use xlinkHref="#pl-icon_vod" />
                </svg>
            </figure>
        ));
        assert.equal(textComponent.text(), translatedString);
    });

    // eslint-disable-next-line max-len
    QUnit.test('for player card prop `onSelect`, augments the function to clear listeners, then emit \'player_rec_select\' tracking event before continuing its functionality', function(assert) {
        const onSelectSpy = sinon.spy();
        const trackEventSpy = sinon.spy();
        const selectedVideo = NORMALIZED_V3_RECOMMENDED_VIDEOS[0];
        const pinnedVODComponent = mountPinnedVOD({
            windowObj: this.windowObj,
            onSelect: onSelectSpy,
            trackEvent: trackEventSpy,
        });
        const cardComponent = pinnedVODComponent.childAt(0).childAt(1);
        const onSelectHandler = cardComponent.prop('onSelect');

        trackEventSpy.reset();
        assert.equal(this.windowObj.clearInterval.callCount, 0);
        assert.equal(onSelectSpy.callCount, 0);

        onSelectHandler(selectedVideo);

        assert.equal(this.windowObj.clearInterval.callCount, 1);
        assert.equal(this.windowObj.clearInterval.firstCall.args[0], this.intervalID);

        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_select');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            /* eslint-disable camelcase */
            recommended_vod_id: selectedVideo.id,
            recommended_vod_type: selectedVideo.recommendationType,
            recommended_vod_view: 'featured',
            /* eslint-enable camelcase */
        });

        assert.equal(onSelectSpy.callCount, 1);
        assert.equal(onSelectSpy.firstCall.args[0], selectedVideo);

        assert.ok(this.windowObj.clearInterval.calledBefore(onSelectSpy));
        assert.ok(trackEventSpy.calledBefore(onSelectSpy));
    });

    QUnit.test('on mount, begins updating featured card\'s progressBarPercentage prop', function(assert) {
        const pinnedVODComponent = mountPinnedVOD({
            windowObj: this.windowObj,
        });
        let cardComponent = pinnedVODComponent.childAt(0).childAt(1);
        const intervalListener = this.windowObj.setInterval.firstCall.args[0];
        const firstIncrement = AUTOPLAY_PERIOD_IN_MS / 4;
        const secondIncrement = AUTOPLAY_PERIOD_IN_MS / 2;
        const firstDateNow = this.startDateNow + firstIncrement;
        const secondDateNow = this.startDateNow + secondIncrement;

        assert.equal(cardComponent.prop('progressBarPercentage'), 100);

        this.windowObj.Date.now = () => firstDateNow;
        intervalListener();
        pinnedVODComponent.update();
        cardComponent = pinnedVODComponent.childAt(0).childAt(1);

        assert.equal(cardComponent.prop('progressBarPercentage'), getPercentage(
            AUTOPLAY_PERIOD_IN_MS - firstIncrement, AUTOPLAY_PERIOD_IN_MS));

        this.windowObj.Date.now = () => secondDateNow;
        intervalListener();
        pinnedVODComponent.update();
        cardComponent = pinnedVODComponent.childAt(0).childAt(1);

        assert.equal(cardComponent.prop('progressBarPercentage'), getPercentage(
            AUTOPLAY_PERIOD_IN_MS - secondIncrement, AUTOPLAY_PERIOD_IN_MS));
    });

    QUnit.test('clears interval listener on dismount', function(assert) {
        const pinnedVODComponent = mountPinnedVOD({
            windowObj: this.windowObj,
        });

        assert.equal(this.windowObj.setInterval.callCount, 1);
        assert.equal(this.windowObj.clearInterval.callCount, 0);

        pinnedVODComponent.unmount();

        assert.equal(this.windowObj.setInterval.callCount, 1);
        assert.equal(this.windowObj.clearInterval.callCount, 1);
        assert.equal(this.windowObj.clearInterval.firstCall.args[0], this.intervalID);
    });

    // eslint-disable-next-line max-len
    QUnit.test('clears interval listener, emits \'player_rec_autoplay\' tracking event, then selects featured video id after autoplay period passes', function(assert) {
        const onSelectSpy = sinon.spy();
        const trackEventSpy = sinon.spy();
        // eslint-disable-next-line no-unused-vars
        const pinnedVODComponent = mountPinnedVOD({
            windowObj: this.windowObj,
            onSelect: onSelectSpy,
            trackEvent: trackEventSpy,
        });
        const intervalListener = this.windowObj.setInterval.firstCall.args[0];
        trackEventSpy.reset();

        assert.equal(onSelectSpy.callCount, 0);
        assert.equal(this.windowObj.clearInterval.callCount, 0);

        this.windowObj.Date.now = () => this.startDateNow + AUTOPLAY_PERIOD_IN_MS + 1;
        intervalListener();

        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_autoplay');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            /* eslint-disable camelcase */
            recommended_vod_id: DEFAULT_PROPS.video.id,
            recommended_vod_type: DEFAULT_PROPS.video.recommendationType,
            /* eslint-enable camelcase */
        });

        assert.equal(onSelectSpy.callCount, 1);
        assert.equal(onSelectSpy.firstCall.args[0], DEFAULT_PROPS.video);

        assert.equal(this.windowObj.clearInterval.callCount, 1);
        assert.ok(this.windowObj.clearInterval.calledBefore(onSelectSpy));
        assert.ok(trackEventSpy.calledBefore(onSelectSpy));
    });

    QUnit.test('emits tracking event \'player_rec_show_featured\' on componentWillMount', function(assert) {
        const trackEventSpy = sinon.spy();
        // eslint-disable-next-line no-unused-vars
        const pinnedVODComponent = shallowRenderPinnedVOD({
            trackEvent: trackEventSpy,
        });

        const instance = pinnedVODComponent.instance();
        trackEventSpy.reset();

        instance.componentWillMount();
        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_show_featured');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            /* eslint-disable camelcase */
            recommended_vod_id: DEFAULT_PROPS.video.id,
            recommended_vod_type: DEFAULT_PROPS.video.recommendationType,
            /* eslint-enable camelcase */
        });
    });
});
