import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PlayerCard } from 'ui/components/recommendations/cards/player-card';
import { AutoplayProgressBar } from 'ui/components/recommendations/cards/progress-bar';
import sinon from 'sinon';
import assign from 'lodash/assign';

const DEFAULT_PROPS = {
    className: 'a classy class act classname',
    thumbnailURL: 'a thumbnail url',
    bread: 'boudin sourdough bread',
    title: 'a titular title',
    i18n: {
        translate: str => str,
    },
    info: 'just some info',
    onSelect: () => {},
    selectItem: {},
    hasThumbZoom: false,
    hasBorder: false,
};

function renderPlayerCard(propOverrides = {}) {
    const props = assign({}, DEFAULT_PROPS, propOverrides);
    const component = <PlayerCard {...props} />;
    return shallow(component);
}

reactTest('ui | components | recommendations | cards | player-card', function() {
    QUnit.test('passes in className and appends pl-card', function(assert) {
        const playerCardElement = renderPlayerCard();
        assert.ok(playerCardElement.hasClass('pl-card'));
        assert.ok(playerCardElement.hasClass(DEFAULT_PROPS.className));
    });

    QUnit.test('adds class when hasThumbZoom is true', function(assert) {
        const playerCardElement = renderPlayerCard({
            hasThumbZoom: true,
        });

        assert.ok(playerCardElement.hasClass('pl-card--thumbzoom'));
    });

    QUnit.test('adds class when hasBorder is true', function(assert) {
        const playerCardElement = renderPlayerCard({
            hasBorder: true,
        });

        assert.ok(playerCardElement.hasClass('pl-card--withborder'));
    });

    QUnit.test('has tabIndex and role prop', function(assert) {
        const playerCardElement = renderPlayerCard();
        assert.equal(playerCardElement.prop('role'), 'button');
        assert.equal(playerCardElement.prop('tabIndex'), 0);
    });

    QUnit.test('has only child as hyperlink with correct class', function(assert) {
        const playerCardElement = renderPlayerCard();
        const hyperLinkElement = playerCardElement.childAt(0);
        assert.equal(playerCardElement.children.length, 1);
        assert.equal(hyperLinkElement.type(), 'a');
        assert.equal(hyperLinkElement.prop('href'), '##');
        assert.ok(hyperLinkElement.hasClass('pl-card__layout'));
    });

    QUnit.test('has correctly populated thumbnail', function(assert) {
        const playerCardElement = renderPlayerCard();
        const thumbnailElement = playerCardElement.childAt(0).childAt(0);
        const imgElement = thumbnailElement.childAt(0);

        assert.equal(thumbnailElement.type(), 'figure');
        assert.ok(thumbnailElement.hasClass('pl-card__img'));

        assert.equal(imgElement.type(), 'img');
        assert.equal(imgElement.prop('alt'), 'Video Thumbnail');
        assert.equal(imgElement.prop('src'), DEFAULT_PROPS.thumbnailURL);
    });

    QUnit.test('has card overlay div', function(assert) {
        const playerCardElement = renderPlayerCard();
        const overlayElement = playerCardElement.childAt(0).childAt(1);

        assert.equal(overlayElement.type(), 'div');
        assert.ok(overlayElement.hasClass('pl-card__overlay'));
    });

    QUnit.test('has card body div', function(assert) {
        const playerCardElement = renderPlayerCard();
        const bodyElement = playerCardElement.childAt(0).childAt(1).childAt(0);

        assert.equal(bodyElement.type(), 'div');
        assert.ok(bodyElement.hasClass('pl-card__body'));
    });

    QUnit.test('has correctly populated bread', function(assert) {
        const playerCardElement = renderPlayerCard();
        // eslint-disable-next-line max-len
        const bodyElement = playerCardElement.childAt(0).childAt(1).childAt(0);
        const breadElement = bodyElement.childAt(0);

        assert.equal(breadElement.type(), 'h4');
        assert.ok(breadElement.hasClass('pl-card__bread'));
        assert.ok(breadElement.hasClass('ellipsis'));
        assert.equal(breadElement.text(), DEFAULT_PROPS.bread);
    });

    QUnit.test('has correctly populated title', function(assert) {
        const playerCardElement = renderPlayerCard();
        // eslint-disable-next-line max-len
        const bodyElement = playerCardElement.childAt(0).childAt(1).childAt(0);
        const titleElement = bodyElement.childAt(1);

        assert.equal(titleElement.type(), 'h3');
        assert.ok(titleElement.hasClass('pl-card__title'));
        assert.ok(titleElement.hasClass('ellipsis'));
        assert.equal(titleElement.text(), DEFAULT_PROPS.title);
    });

    QUnit.test('has correctly populated info', function(assert) {
        const playerCardElement = renderPlayerCard();
        // eslint-disable-next-line max-len
        const bodyElement = playerCardElement.childAt(0).childAt(1).childAt(0);
        const infoElement = bodyElement.childAt(2);

        assert.equal(infoElement.type(), 'div');
        assert.ok(infoElement.hasClass('pl-card__info'));
        assert.ok(infoElement.hasClass('ellipsis'));
        assert.equal(infoElement.text(), DEFAULT_PROPS.info);
    });

    QUnit.test('attaches onClick handler that calls onSelect and passes selectItem in', function(assert) {
        const selectHandler = sinon.spy();
        const selectItem = {
            item: 'someItem',
        };

        const playerCardComponent = renderPlayerCard({
            onSelect: selectHandler,
            selectItem,
        });

        playerCardComponent.simulate('click');
        assert.equal(selectHandler.callCount, 1);
        assert.equal(selectHandler.firstCall.args[0], selectItem);
    });

    QUnit.test('attaches onKeyUp `Enter` handler that calls onSelect and passes selectItem in', function(assert) {
        const selectHandler = sinon.spy();
        const selectItem = {
            item: 'someItem',
        };

        const playerCardComponent = renderPlayerCard({
            onSelect: selectHandler,
            selectItem,
        });

        playerCardComponent.simulate('keyup', { key: 'Shift' });
        assert.equal(selectHandler.callCount, 0);
        playerCardComponent.simulate('keyup', { key: 'Enter' });
        assert.equal(selectHandler.callCount, 1);
        assert.equal(selectHandler.firstCall.args[0], selectItem);
    });

    // eslint-disable-next-line max-len
    QUnit.test('if there is a progressBarPercentage prop, add a class and a AutoplayProgressBar component', function(assert) {
        const validProgressBarPercentage = 50;
        const playerCardElement = renderPlayerCard({
            progressBarPercentage: validProgressBarPercentage,
        });

        assert.ok(playerCardElement.hasClass('pl-card--withprogress'));

        // eslint-disable-next-line max-len
        const overlayElement = playerCardElement.childAt(0).childAt(1);
        const progressBarElement = overlayElement.childAt(1);

        assert.equal(progressBarElement.type(), AutoplayProgressBar);
        assert.equal(progressBarElement.prop('percentage'), validProgressBarPercentage);
    });

    // eslint-disable-next-line max-len
    QUnit.test('if there is not a progressBarPercentage prop, dont add a AutoplayProgressBar component', function(assert) {
        const playerCardElement = renderPlayerCard();
        // eslint-disable-next-line max-len
        const overlayElement = playerCardElement.childAt(0).childAt(1);

        assert.equal(overlayElement.children().length, 1);
    });
});
