import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { OfflineRecommendations } from 'ui/components/recommendations/offline-recommendations';
import { PinnedVOD } from 'ui/components/recommendations/pinned-vod';
import { OtherVODs } from 'ui/components/recommendations/other-vods';
import { NORMALIZED_V5_RECOMMENDED_VIDEOS } from 'tests/fixtures/recommendations';
import sinon from 'sinon';

const DEFAULT_PROPS = Object.freeze({
    videos: NORMALIZED_V5_RECOMMENDED_VIDEOS,
    i18n: {},
    onSelectVOD: () => {},
    playerHeight: 100,
    playerWidth: 100,
    onExit: () => {},
    trackEvent: () => {},
    streamWasOnline: false,
});

function shallowRenderRecommendations(overrides = {}) {
    const args = assign({}, DEFAULT_PROPS, overrides);
    const component = <OfflineRecommendations {...args} />;
    return shallow(component);
}

reactTest('ui | components | recommendations | offline-recommendations', function() {
    QUnit.test('has correct classes on main div when showing featured vod', function(assert) {
        const recComponent = shallowRenderRecommendations().setState({
            showFeatured: true,
        });

        assert.equal(recComponent.type(), 'div');
        assert.ok(recComponent.hasClass('player-overlay'));
        assert.ok(recComponent.hasClass('recommendations-overlay'));
        assert.notOk(recComponent.hasClass('pl-rec-wrapper'));
    });

    QUnit.test('has correct classes on main div when not showing featured vod', function(assert) {
        const recComponent = shallowRenderRecommendations().setState({
            showFeatured: false,
        });

        assert.equal(recComponent.type(), 'div');
        assert.ok(recComponent.hasClass('player-overlay'));
        assert.ok(recComponent.hasClass('recommendations-overlay'));
        assert.ok(recComponent.hasClass('pl-rec-wrapper'));
    });

    QUnit.test('on mount shows featured vod pane first', function(assert) {
        const recComponent = shallowRenderRecommendations();

        assert.equal(recComponent.children().length, 1);
        assert.ok(recComponent.childAt(0).containsMatchingElement(
            <PinnedVOD
                onSelect={DEFAULT_PROPS.onSelectVOD}
                i18n={DEFAULT_PROPS.i18n}
                width={DEFAULT_PROPS.playerWidth}
                video={DEFAULT_PROPS.videos[0]}
                trackEvent={DEFAULT_PROPS.trackEvent}
            />
        ));
    });

    QUnit.test('should show other vods pane, when the featured vod pane calls its onClose prop', function(assert) {
        const recComponent = shallowRenderRecommendations();

        assert.equal(recComponent.childAt(0).type(), PinnedVOD);

        const onCloseFunction = recComponent.childAt(0).prop('onClose');
        onCloseFunction();

        recComponent.update();

        assert.ok(recComponent.childAt(0).containsMatchingElement(
            <OtherVODs
                onSelect={DEFAULT_PROPS.onSelectVOD}
                width={DEFAULT_PROPS.playerWidth}
                height={DEFAULT_PROPS.playerHeight}
                videos={DEFAULT_PROPS.videos}
                onClose={DEFAULT_PROPS.onExit}
                trackEvent={DEFAULT_PROPS.trackEvent}
                i18n={DEFAULT_PROPS.i18n}
                windowObj={DEFAULT_PROPS.windowObj}
            />
        ));
    });

    // eslint-disable-next-line max-len
    QUnit.test('emits tracking event \'player_rec_show\' with offline type on componentWillMount when shown before stream has played', function(assert) {
        const trackEventSpy = sinon.spy();
        // eslint-disable-next-line no-unused-vars
        const recComponent = shallowRenderRecommendations({
            trackEvent: trackEventSpy,
            streamWasOnline: false,
        });

        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_show');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            type: 'offline',
        });
    });

    // eslint-disable-next-line max-len
    QUnit.test('emits tracking event \'player_rec_show\' with postLive type on componentWillMount when shown before stream has played', function(assert) {
        const trackEventSpy = sinon.spy();
        // eslint-disable-next-line no-unused-vars
        const recComponent = shallowRenderRecommendations({
            trackEvent: trackEventSpy,
            streamWasOnline: true,
        });

        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_show');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            type: 'postLive',
        });
    });
});
