import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PlayerCard } from 'ui/components/recommendations/cards/player-card';
import { RecCloseButton } from 'ui/components/buttons/rec-close-button';
import { InfoDateLength } from 'ui/components/recommendations/cards/info-date-length';
import { OtherVODsComponent, RECOMMENDATION_COLUMN_CONFIG,
         RECOMMENDATION_ROW_CONFIG } from 'ui/components/recommendations/other-vods';
import { NORMALIZED_V3_RECOMMENDED_VIDEOS } from 'tests/fixtures/recommendations';
import { CHANNEL_VODS, SIMILAR_VODS } from 'actions/recommendations';
import assign from 'lodash/assign';
import find from 'lodash/find';
import sinon from 'sinon';

const DEFAULT_PROPS = Object.freeze({
    width: 0,
    height: 0,
    videos: NORMALIZED_V3_RECOMMENDED_VIDEOS,
    onSelect: () => {},
    i18n: {
        langCode: 'languageCode',
    },
    windowObj: {
        Date: property => ({
            dateObjectProperty: property,
        }),

    },
    trackEvent: () => {},
    onClose: () => {},
    t: str => str,
});

function shallowRenderPaneWrapper(propOverrides = {}) {
    const props = assign({}, DEFAULT_PROPS, propOverrides);
    const component = <OtherVODsComponent {...props} />;
    return shallow(component);
}

reactTest('ui | components | recommendations | other-vods', function() {
    QUnit.test('has correct classes', function(assert) {
        const paneComponent = shallowRenderPaneWrapper();
        assert.ok(paneComponent.hasClass('pl-rec'));
        assert.ok(paneComponent.hasClass('pl-flex'));
        assert.ok(paneComponent.hasClass('pl-flex--stretch'));
        assert.ok(paneComponent.hasClass('pl-flex--horizontalCenter'));
    });

    QUnit.test('has a close button', function(assert) {
        const paneComponent = shallowRenderPaneWrapper();
        const buttonComponent = paneComponent.childAt(0);

        assert.equal(buttonComponent.type(), RecCloseButton);
        assert.equal(buttonComponent.prop('onClick'), DEFAULT_PROPS.onClose);
    });

    QUnit.test('has `More from ${channelName}` translated title if channel videos', function(assert) {
        const channelVideos = NORMALIZED_V3_RECOMMENDED_VIDEOS.map(v => assign({}, v, {
            recommendationType: CHANNEL_VODS,
        }));
        const translatedTitle = 'a translated title';
        const translateStub = sinon.stub();
        translateStub.withArgs(
            'More from {{channelName}}',
            { channelName: channelVideos[0].channelName }
        ).returns(translatedTitle);
        const paneComponent = shallowRenderPaneWrapper({
            t: translateStub,
            videos: channelVideos,
        });
        const titleComponent = paneComponent.childAt(1);

        assert.equal(titleComponent.type(), 'h2');
        assert.ok(titleComponent.hasClass('pl-rec__title'));
        assert.ok(titleComponent.hasClass('pl-flex__item'));
        assert.equal(titleComponent.text(), translatedTitle);
    });

    QUnit.test('has `Watch Now` translated title if similar videos', function(assert) {
        const similarVideos = NORMALIZED_V3_RECOMMENDED_VIDEOS.map(v => assign({}, v, {
            recommendationType: SIMILAR_VODS,
        }));
        const translatedTitle = 'a translated title';
        const translateStub = sinon.stub();
        translateStub.withArgs('Watch Now').returns(translatedTitle);
        const paneComponent = shallowRenderPaneWrapper({
            t: translateStub,
            videos: similarVideos,
        });
        const titleComponent = paneComponent.childAt(1);

        assert.equal(titleComponent.type(), 'h2');
        assert.ok(titleComponent.hasClass('pl-rec__title'));
        assert.ok(titleComponent.hasClass('pl-flex__item'));
        assert.equal(titleComponent.text(), translatedTitle);
    });

    QUnit.test('has a container for all its player cards', function(assert) {
        const paneComponent = shallowRenderPaneWrapper();
        const cardContainerComponent = paneComponent.childAt(2);

        assert.equal(cardContainerComponent.type(), 'div');
        assert.ok(cardContainerComponent.hasClass('pl-rec__container'));
        assert.ok(cardContainerComponent.hasClass('pl-flex'));
        assert.ok(cardContainerComponent.hasClass('pl-flex__item'));
        assert.ok(cardContainerComponent.hasClass('pl-flex--verticalCenter'));
        assert.ok(cardContainerComponent.hasClass('pl-flex--horizontalCenter'));
    });

    // eslint-disable-next-line max-len
    QUnit.test('the player card container, for each video of recommendationType `channel`, creates a correct player card', function(assert) {
        const channelVideos = NORMALIZED_V3_RECOMMENDED_VIDEOS.map(v => assign({}, v, {
            recommendationType: CHANNEL_VODS,
        }));
        const paneComponent = shallowRenderPaneWrapper({
            videos: channelVideos,
        });
        const cardComponents = paneComponent.childAt(2).children();

        assert.equal(cardComponents.length, channelVideos.length);

        cardComponents.forEach((cardComponent, index) => {
            assert.ok(cardComponent.containsMatchingElement(
                <PlayerCard
                    thumbnailURL={channelVideos[index].thumbnailURL}
                    title={channelVideos[index].title}
                    info={channelVideos[index].game}
                    selectItem={channelVideos[index]}
                    hasThumbZoom={true}
                />
            ));
        });
    });

    // eslint-disable-next-line max-len
    QUnit.test('the player card container, for each video of recommendationType `similar`, creates a correct player card', function(assert) {
        const similarVideos = NORMALIZED_V3_RECOMMENDED_VIDEOS.map(v => assign({}, v, {
            recommendationType: SIMILAR_VODS,
        }));
        const paneComponent = shallowRenderPaneWrapper({
            videos: similarVideos,
        });
        const cardComponents = paneComponent.childAt(2).children();

        assert.equal(cardComponents.length, DEFAULT_PROPS.videos.length);

        cardComponents.forEach((cardComponent, index) => {
            assert.ok(cardComponent.containsMatchingElement(
                <PlayerCard
                    thumbnailURL={similarVideos[index].thumbnailURL}
                    bread={similarVideos[index].channelName}
                    title={similarVideos[index].title}
                    info={
                        <InfoDateLength
                            length={similarVideos[index].duration}
                            date={new DEFAULT_PROPS.windowObj.Date(similarVideos[index].creationDate)}
                            languageCode={DEFAULT_PROPS.i18n.langCode}
                        />
                    }
                    selectItem={similarVideos[index]}
                    hasThumbZoom={true}
                />
            ));
        });
    });

    QUnit.test('augments card\'s onSelect prop with tracking event \'player_rec_select\'', function(assert) {
        const selectedVideo = NORMALIZED_V3_RECOMMENDED_VIDEOS[0];
        const trackEventSpy = sinon.spy();
        const onSelectSpy = sinon.spy();
        const paneComponent = shallowRenderPaneWrapper({
            trackEvent: trackEventSpy,
            onSelect: onSelectSpy,
        });
        const cardComponents = paneComponent.childAt(2).children();

        cardComponents.forEach(cardComponent => {
            trackEventSpy.reset();
            onSelectSpy.reset();
            cardComponent.prop('onSelect')(selectedVideo);

            assert.equal(trackEventSpy.callCount, 1);
            assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_select');
            assert.deepEqual(trackEventSpy.firstCall.args[1], {
                /* eslint-disable camelcase */
                recommended_vod_id: selectedVideo.id,
                recommended_vod_type: selectedVideo.recommendationType,
                recommended_vod_view: 'other',
                /* eslint-enable camelcase */
            });
        });
    });

    QUnit.test('emits tracking event \'player_rec_show_others\' on componentWillMount', function(assert) {
        const width = 780;
        const height = 670;
        const { columns } = find(RECOMMENDATION_COLUMN_CONFIG, cfg => width >= cfg.minPlayerWidth);
        const { rows } = find(RECOMMENDATION_ROW_CONFIG, cfg => height >= cfg.minPlayerHeight);
        const numOfCardsVisible = columns * rows;

        const concatenatedVODIDs = DEFAULT_PROPS.videos.
            slice(0, numOfCardsVisible).
            map(v => v.id).
            join(',');

        const trackEventSpy = sinon.spy();
        // eslint-disable-next-line no-unused-vars
        const paneComponent = shallowRenderPaneWrapper({
            width,
            height,
            trackEvent: trackEventSpy,
        });

        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_show_others');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            /* eslint-disable camelcase */
            recommended_vod_ids: concatenatedVODIDs,
            recommended_vod_types: DEFAULT_PROPS.videos[0].recommendationType,
            /* eslint-enable camelcase */
        });
    });

    QUnit.module('using hard coded configs with height and width set', function(hooks) {
        hooks.beforeEach(function() {
            const width = 780;
            const height = 670;

            const { columns } = find(RECOMMENDATION_COLUMN_CONFIG, cfg => width >= cfg.minPlayerWidth);
            const { rows } = find(RECOMMENDATION_ROW_CONFIG, cfg => height >= cfg.minPlayerHeight);
            this.numOfCardsVisible = columns * rows;

            this.trackEventSpy = sinon.spy();

            this.paneComponent = shallowRenderPaneWrapper({
                width,
                height,
                trackEvent: this.trackEventSpy,
            });
        });

        QUnit.test('emits tracking event \'player_rec_show_others\' on componentWillMount', function(assert) {
            const concatenatedVODIDs = DEFAULT_PROPS.videos.
                slice(0, this.numOfCardsVisible).
                map(v => v.id).
                join(',');

            assert.equal(this.trackEventSpy.callCount, 1);
            assert.equal(this.trackEventSpy.firstCall.args[0], 'player_rec_show_others');
            assert.deepEqual(this.trackEventSpy.firstCall.args[1], {
                /* eslint-disable camelcase */
                recommended_vod_ids: concatenatedVODIDs,
                recommended_vod_types: DEFAULT_PROPS.videos[0].recommendationType,
                /* eslint-enable camelcase */
            });
        });

        QUnit.test('displays the correct number of visible player cards', function(assert) {
            this.paneComponent.childAt(2).children().forEach((cardComponent, index) => {
                if (index >= this.numOfCardsVisible) {
                    assert.ok(cardComponent.hasClass('hidden'));
                } else {
                    assert.notOk(cardComponent.hasClass('hidden'));
                }
            });
        });

        QUnit.test('sets key of player cards based on video id and visibility', function(assert) {
            this.paneComponent.childAt(2).children().forEach((cardComponent, index) => {
                const expectedKey = `${DEFAULT_PROPS.videos[index].id}_${index >= this.numOfCardsVisible}`;
                assert.equal(cardComponent.key(), expectedKey);
            });
        });
    });
});
