import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PinnedVODComponent } from 'ui/components/recommendations/pinned-vod';
import { PlayerCard } from 'ui/components/recommendations/cards/player-card';
import { RecCloseButton } from 'ui/components/buttons/rec-close-button';
import { NORMALIZED_V3_RECOMMENDED_VIDEOS } from 'tests/fixtures/recommendations';
import { FeaturedCollection } from 'ui/containers/collections/featured-collection';
import assign from 'lodash/assign';
import sinon from 'sinon';

const DEFAULT_PROPS = {
    video: NORMALIZED_V3_RECOMMENDED_VIDEOS[0],
    onSelect: () => {},
    onClose: () => {},
    width: 0,
    t: () => 'a string',
    trackEvent: () => {},
};

function shallowRenderPinnedVOD(propOverrides = {}) {
    const props = assign({}, DEFAULT_PROPS, propOverrides);
    const component = <PinnedVODComponent {...props} />;
    return shallow(component);
}

reactTest('ui | components | recommendations | pinned-vod', function() {
    QUnit.test('has correct class', function(assert) {
        const pinnedVODComponent = shallowRenderPinnedVOD();
        assert.ok(pinnedVODComponent.hasClass('pl-pinned-item'));
    });

    QUnit.test('has pinned item title as the first child', function(assert) {
        const pinnedVODComponent = shallowRenderPinnedVOD();
        const pinnedItemTitleComponent = pinnedVODComponent.childAt(0);

        assert.ok(pinnedItemTitleComponent.hasClass('pl-flex'));
        assert.ok(pinnedItemTitleComponent.hasClass('pl-pinned-item__title'));
    });

    QUnit.test('the pinned item title text is properly translated', function(assert) {
        const translatedTitleText = 'a correctly translated string';
        const translateStub = sinon.stub();
        translateStub.withArgs('Watch Now').returns(translatedTitleText);
        const pinnedVODComponent = shallowRenderPinnedVOD({
            t: translateStub,
        });

        const pinnedItemTitleTextComponent = pinnedVODComponent.childAt(0).childAt(0);

        assert.ok(pinnedItemTitleTextComponent.hasClass('pl-flex__item'));
        assert.ok(pinnedItemTitleTextComponent.hasClass('pl-flex__item--grow'));
        assert.equal(pinnedItemTitleTextComponent.type(), 'span');
        assert.equal(pinnedItemTitleTextComponent.text(), translatedTitleText);
    });

    QUnit.test('the pinned item title has a close button', function(assert) {
        const pinnedVODComponent = shallowRenderPinnedVOD();
        const closeButtonComponent = pinnedVODComponent.childAt(0).childAt(1);
        assert.equal(closeButtonComponent.type(), RecCloseButton);
        assert.equal(closeButtonComponent.prop('onClick'), DEFAULT_PROPS.onClose);
    });

    QUnit.test('has correctly populated player card as the second child', function(assert) {
        const pinnedVODComponent = shallowRenderPinnedVOD();
        const cardComponent = pinnedVODComponent.childAt(1);

        assert.ok(cardComponent.containsMatchingElement(
            <PlayerCard
                className="pl-rec__item"
                thumbnailURL={DEFAULT_PROPS.video.thumbnailURL}
                selectItem={DEFAULT_PROPS.video}
                hasThumbZoom={true}
                hasBorder={true}
            />
        ));
    });

    QUnit.test('has a featured collection card experiment as the third child', function(assert) {
        const pinnedVODComponent = shallowRenderPinnedVOD();
        const featuredCollection = pinnedVODComponent.childAt(2);

        assert.equal(featuredCollection.type(), FeaturedCollection);
    });

    QUnit.test('for player card prop `info`, has VOD SVG and correctly translated string', function(assert) {
        const translatedString = 'a translated string';
        const translateStub = sinon.stub().withArgs('Most Recent Video').returns(translatedString);
        const pinnedVODComponent = shallowRenderPinnedVOD({
            t: translateStub,
        });

        const cardComponent = pinnedVODComponent.childAt(1);
        const cardInfoComponent = shallow(cardComponent.prop('info'));
        const VODSVGComponent = cardInfoComponent.childAt(0);
        const textComponent = cardInfoComponent.childAt(1);

        assert.equal(cardInfoComponent.type(), 'span');
        assert.ok(VODSVGComponent.containsMatchingElement(
            <figure className="pl-card__smallicon">
                <svg>
                    <use xlinkHref="#pl-icon_vod" />
                </svg>
            </figure>
        ));
        assert.equal(textComponent.text(), translatedString);
    });

    // eslint-disable-next-line max-len
    QUnit.test('for player card prop `onSelect`, emit \'player_rec_select\' tracking event before calling onSelect', function(assert) {
        const onSelectSpy = sinon.spy();
        const trackEventSpy = sinon.spy();
        const selectedVideo = NORMALIZED_V3_RECOMMENDED_VIDEOS[0];
        const pinnedVODComponent = shallowRenderPinnedVOD({
            onSelect: onSelectSpy,
            trackEvent: trackEventSpy,
        });
        const cardComponent = pinnedVODComponent.childAt(1);
        const onSelectHandler = cardComponent.prop('onSelect');

        trackEventSpy.reset();
        assert.equal(onSelectSpy.callCount, 0);

        onSelectHandler(selectedVideo);

        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_select');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            /* eslint-disable camelcase */
            recommended_vod_id: selectedVideo.id,
            recommended_vod_type: selectedVideo.recommendationType,
            recommended_vod_view: 'featured',
            /* eslint-enable camelcase */
        });

        assert.equal(onSelectSpy.callCount, 1);
        assert.equal(onSelectSpy.firstCall.args[0], selectedVideo);
        assert.ok(trackEventSpy.calledBefore(onSelectSpy));
    });

    QUnit.test('emits tracking event \'player_rec_show_featured\' on componentWillMount', function(assert) {
        const trackEventSpy = sinon.spy();
        // eslint-disable-next-line no-unused-vars
        const pinnedVODComponent = shallowRenderPinnedVOD({
            trackEvent: trackEventSpy,
        });

        const instance = pinnedVODComponent.instance();
        trackEventSpy.reset();

        instance.componentWillMount();
        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_show_featured');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            /* eslint-disable camelcase */
            recommended_vod_id: DEFAULT_PROPS.video.id,
            recommended_vod_type: DEFAULT_PROPS.video.recommendationType,
            /* eslint-enable camelcase */
        });
    });
});
