import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PostVODRecommendations } from 'ui/components/recommendations/post-vod-recommendations';
import { AutoplayingPinnedVOD } from 'ui/components/recommendations/autoplaying-pinned-vod';
import { OtherVODs } from 'ui/components/recommendations/other-vods';
import { NORMALIZED_V3_RECOMMENDED_VIDEOS } from 'tests/fixtures/recommendations';
import sinon from 'sinon';

const DEFAULT_PROPS = Object.freeze({
    videos: NORMALIZED_V3_RECOMMENDED_VIDEOS,
    i18n: {},
    onSelectVOD: () => {},
    playerHeight: 100,
    playerWidth: 100,
    onExit: () => {},
    trackEvent: () => {},
    windowObj: {},
    isMiniPlayer: false,
});

function shallowRenderRecommendations(overrides = {}) {
    const args = assign({}, DEFAULT_PROPS, overrides);
    const component = <PostVODRecommendations {...args} />;
    return shallow(component);
}

reactTest('ui | components | recommendations | post-vod-recommendations', function() {
    QUnit.test('has correct classes on main div', function(assert) {
        const recComponent = shallowRenderRecommendations();

        assert.equal(recComponent.type(), 'div');
        assert.ok(recComponent.hasClass('player-overlay'));
        assert.ok(recComponent.hasClass('recommendations-overlay'));
        assert.ok(recComponent.hasClass('pl-rec-wrapper'));
    });

    QUnit.test('on mount renders autoplaying vod if isMiniPlayer is false', function(assert) {
        const recComponent = shallowRenderRecommendations({
            isMiniPlayer: false,
        });

        assert.equal(recComponent.children().length, 1);
        assert.ok(recComponent.childAt(0).containsMatchingElement(
            <AutoplayingPinnedVOD
                onSelect={DEFAULT_PROPS.onSelectVOD}
                i18n={DEFAULT_PROPS.i18n}
                width={DEFAULT_PROPS.playerWidth}
                video={DEFAULT_PROPS.videos[0]}
                trackEvent={DEFAULT_PROPS.trackEvent}
                windowObj={DEFAULT_PROPS.windowObj}
            />
        ), 'autoplaying vod properly rendered');
    });

    QUnit.test('renders nothing if isMiniPlayer is true', function(assert) {
        const recComponent = shallowRenderRecommendations({
            isMiniPlayer: true,
        });

        assert.equal(recComponent.type(), null);
    });

    // eslint-disable-next-line max-len
    QUnit.test('if mounted while isMiniPlayer is true, shows other vods pane once isMiniPlayer is false', function(assert) {
        const recComponent = shallowRenderRecommendations({
            isMiniPlayer: true,
        });

        assert.equal(recComponent.type(), null);

        recComponent.setProps({
            isMiniPlayer: false,
        });

        assert.equal(recComponent.children().length, 1);
        assert.ok(recComponent.childAt(0).containsMatchingElement(
            <OtherVODs
                onSelect={DEFAULT_PROPS.onSelectVOD}
                width={DEFAULT_PROPS.playerWidth}
                height={DEFAULT_PROPS.playerHeight}
                videos={DEFAULT_PROPS.videos}
                onClose={DEFAULT_PROPS.onExit}
                trackEvent={DEFAULT_PROPS.trackEvent}
                i18n={DEFAULT_PROPS.i18n}
                windowObj={DEFAULT_PROPS.windowObj}
            />
        ), 'other vods properly rendered');
    });

    QUnit.test('should render other vods, when the autoplaying vod calls its onClose prop', function(assert) {
        const recComponent = shallowRenderRecommendations();

        assert.equal(recComponent.childAt(0).type(), AutoplayingPinnedVOD);

        const onCloseFunction = recComponent.childAt(0).prop('onClose');
        onCloseFunction();

        recComponent.update();

        assert.ok(recComponent.childAt(0).containsMatchingElement(
            <OtherVODs
                onSelect={DEFAULT_PROPS.onSelectVOD}
                width={DEFAULT_PROPS.playerWidth}
                height={DEFAULT_PROPS.playerHeight}
                videos={DEFAULT_PROPS.videos}
                onClose={DEFAULT_PROPS.onExit}
                trackEvent={DEFAULT_PROPS.trackEvent}
                i18n={DEFAULT_PROPS.i18n}
                windowObj={DEFAULT_PROPS.windowObj}
            />
        ), 'other vods properly rendered');
    });

    QUnit.test('emits tracking event \'player_rec_show\' on componentWillMount', function(assert) {
        const trackEventSpy = sinon.spy();
        // eslint-disable-next-line no-unused-vars
        const recComponent = shallowRenderRecommendations({
            trackEvent: trackEventSpy,
        });

        assert.equal(trackEventSpy.callCount, 1);
        assert.equal(trackEventSpy.firstCall.args[0], 'player_rec_show');
        assert.deepEqual(trackEventSpy.firstCall.args[1], {
            type: 'postVOD',
        });
    });
});
