import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { TEST_PREVIEW_POPUP_OBJECT } from 'tests/fixtures/timeline-metadata';
import { SeekbarPopup, ARROW_WIDTH } from 'ui/components/seekbar/seekbar-popup';

const POPUP_WIDTH = 148;

const DEFAULT_ARGS = Object.freeze({
    popupLeftOffset: 30,
    popupObject: TEST_PREVIEW_POPUP_OBJECT,
    popupWidth: POPUP_WIDTH,
    seekbarWidth: 100,
});

function getLeftOffset(centerX, elementWidth, seekbarWidth) {
    const offset = centerX - (0.5 * elementWidth);
    const leftOffset = Math.max(0, Math.min(offset, seekbarWidth - elementWidth));

    return !isNaN(leftOffset) ? leftOffset : 0;
}

function renderSeekbarPopup(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <SeekbarPopup {...args} />;
    return shallow(component);
}

function createThumbnailStyles({ height, width, sheetWidth, url, x, y }) {
    return {
        height: height,
        width: width,
        backgroundSize: `${sheetWidth}px`,
        backgroundImage: `url(${url})`,
        backgroundPosition: `-${x}px -${y}px`,
    };
}

reactTest('ui | components | seekbar-popup', function() {
    QUnit.test('expected title should render', function(assert) {
        const title = QUnit.config.current.testId;
        const popupObject = assign({}, TEST_PREVIEW_POPUP_OBJECT, {
            title,
        });
        const component = renderSeekbarPopup({ popupObject });
        const titleSpan = component.find('span.popup-title');
        assert.equal(titleSpan.type(), 'span', 'title span should exist');
        assert.equal(titleSpan.text(), title, `should render ${title}`);
    });

    QUnit.test('expected timestamp should render', function(assert) {
        const timestamp = TEST_PREVIEW_POPUP_OBJECT.timestamp;
        const component = renderSeekbarPopup();
        const timestampSpan = component.find('span.popup-timestamp');
        assert.equal(timestampSpan.type(), 'span', 'timestamp span should exist');
        assert.equal(timestampSpan.text(), timestamp, `should render ${timestamp}`);
    });

    QUnit.test('positions the popup container correctly', function(assert) {
        const component = renderSeekbarPopup();
        const { seekbarWidth, popupLeftOffset } = DEFAULT_ARGS;
        const timestampDiv = component.find('div.player-slider__popup-container');

        const expectedLeftOffset = getLeftOffset(popupLeftOffset, POPUP_WIDTH, seekbarWidth);

        const { style } = timestampDiv.props();
        assert.deepEqual(style, {
            left: expectedLeftOffset,
            width: POPUP_WIDTH,
        });
    });

    QUnit.test('positions the arrow correctly', function(assert) {
        const component = renderSeekbarPopup();
        const { seekbarWidth, popupLeftOffset } = DEFAULT_ARGS;
        const timestampDiv = component.find('div.popup-arrow');

        const expectedLeftOffset = getLeftOffset(popupLeftOffset, ARROW_WIDTH, seekbarWidth);

        const { style } = timestampDiv.props();
        assert.deepEqual(style, {
            left: expectedLeftOffset,
        });
    });

    QUnit.test('popup-thumb-wraper div should be styled properly', function(assert) {
        const component = renderSeekbarPopup();
        const thumbwrapperDiv = component.find('div.popup-thumb-wrapper');
        const { style } = thumbwrapperDiv.props();
        const expectedStyle = {
            width: TEST_PREVIEW_POPUP_OBJECT.thumbOver.width,
            height: TEST_PREVIEW_POPUP_OBJECT.thumbOver.height,
            maxHeight: TEST_PREVIEW_POPUP_OBJECT.thumbOver.height,
        };

        assert.deepEqual(style, expectedStyle);
    });

    function testThumbnailStyles(thumbObj, className) {
        QUnit.test(`${className} div should be properly styled`, function(assert) {
            const component = renderSeekbarPopup();
            const thumbDiv = component.find(`div.${className}`);
            const { style } = thumbDiv.props();

            assert.deepEqual(style, createThumbnailStyles(thumbObj));
        });
    }

    testThumbnailStyles(TEST_PREVIEW_POPUP_OBJECT.thumbUnder, 'popup-marker-thumbunder');
    testThumbnailStyles(TEST_PREVIEW_POPUP_OBJECT.thumbOver, 'popup-marker-thumbover');
});
