import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import assign from 'lodash/assign';
import * as Timestamp from 'util/timestamp';
import { TimestampPreview } from 'ui/components/seekbar/timestamp-preview';

const DEFAULT_ARGS = Object.freeze({
    duration: 1000,
    mouseMoveClientX: 300,
    seekbarWidth: 1000,
    seekbarLeftOffset: 0,
});

const DEFAULT_POPUP_LEFT_OFFSET = DEFAULT_ARGS.mouseMoveClientX - DEFAULT_ARGS.seekbarLeftOffset;
const ARROW_WIDTH = 20;
const POPUP_WIDTH = 20;

function renderTimestampPreview(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <TimestampPreview {...args} />;
    return shallow(component);
}

function getLeftOffset(centerX, elementWidth, seekbarWidth) {
    const offset = centerX - (0.5 * elementWidth);

    return Math.max(0, Math.min(offset, seekbarWidth - elementWidth));
}

reactTest('ui | components | seekbar | timestamp-preview', function() {
    QUnit.test('creates timestamp preview with correct time', function(assert) {
        const { duration, seekbarWidth } = DEFAULT_ARGS;

        const component = renderTimestampPreview();

        const expectedTimestamp = Timestamp.toString(duration * (DEFAULT_POPUP_LEFT_OFFSET / seekbarWidth), false);
        const actualTimestamp = component.text();

        assert.equal(expectedTimestamp, actualTimestamp);
    });

    QUnit.test('positions the timestamp correctly', function(assert) {
        const component = renderTimestampPreview();
        const { seekbarWidth } = DEFAULT_ARGS;
        const timestampDiv = component.find('div.player-slider__popup-container');

        const expectedLeftOffset = getLeftOffset(DEFAULT_POPUP_LEFT_OFFSET, POPUP_WIDTH, seekbarWidth);

        const { style } = timestampDiv.props();
        assert.deepEqual(style, {
            left: expectedLeftOffset,
            width: POPUP_WIDTH,
        });
    });

    QUnit.test('positions the arrow correctly', function(assert) {
        const component = renderTimestampPreview();
        const { seekbarWidth } = DEFAULT_ARGS;
        const timestampDiv = component.find('div.popup-arrow');

        const expectedLeftOffset = getLeftOffset(DEFAULT_POPUP_LEFT_OFFSET, ARROW_WIDTH, seekbarWidth);

        const { style } = timestampDiv.props();
        assert.deepEqual(style, {
            left: expectedLeftOffset,
        });
    });
});
