import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { AdvancedMenuComponent } from 'ui/components/settings/advanced-menu';
import { IDENTIFIER_MAIN_MENU } from 'ui/components/settings/menu-manager';
import { Toggle } from 'ui/components/common/toggle';

const DEFAULT_ARGS = Object.freeze({
    html5PlayerEnabled: false,
    html5ToggleEnabled: false,
    isShowingVideoStats: false,
    miniPlayerEnabled: false,
    miniPlayerToggleEnabled: false,
    onHTML5Toggle() {},
    onMenuTransition() {},
    onMiniPlayerToggle() {},
    t: str => str,
});

function renderAdvancedMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <AdvancedMenuComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | settings | advanced-menu', function() {
    QUnit.test('returns a div', function(assert) {
        const component = renderAdvancedMenu();
        assert.equal(component.type(), 'div');
    });

    QUnit.test('when the header is clicked, menu transition to main menu is invoked', function(assert) {
        const transitionSpy = sinon.spy();
        const component = renderAdvancedMenu({
            onMenuTransition: transitionSpy,
        });

        component.find('.qa-header-button').simulate('click');
        assert.equal(transitionSpy.callCount, 1, 'transition prop was invoked');
        assert.equal(transitionSpy.firstCall.args[0], IDENTIFIER_MAIN_MENU, 'transition identifier is main menu');
    });

    QUnit.test('if html5ToggleEnabled is true, show html5 toggle', function(assert) {
        const component = renderAdvancedMenu({
            html5ToggleEnabled: true,
        });

        assert.equal(component.find('.qa-html5-toggle').length, 1);
    });

    QUnit.test('if html5ToggleEnabled is false, do not show html5 toggle', function(assert) {
        const component = renderAdvancedMenu({
            html5ToggleEnabled: false,
        });

        assert.equal(component.find('.qa-html5-toggle').length, 0);
    });

    QUnit.test('if miniPlayerToggleEnabled is true, show miniplayer toggle', function(assert) {
        const component = renderAdvancedMenu({
            miniPlayerToggleEnabled: true,
        });

        assert.equal(component.find('.qa-miniplayer-toggle').length, 1);
    });

    QUnit.test('if miniPlayerToggleEnabled is false, do not show miniplayer toggle', function(assert) {
        const component = renderAdvancedMenu({
            miniPlayerToggleEnabled: false,
        });

        assert.equal(component.find('.qa-miniplayer-toggle').length, 0);
    });

    QUnit.test('on toggling html5, invoke onHTML5Toggle prop', function(assert) {
        const html5ToggleSpy = sinon.spy();
        const component = renderAdvancedMenu({
            html5ToggleEnabled: true,
            onHTML5Toggle: html5ToggleSpy,
        });

        component.find(Toggle).props().onToggle();
        assert.equal(html5ToggleSpy.callCount, 1);
    });

    QUnit.test('on toggling miniplayer, invoke onMiniPlayerToggle prop', function(assert) {
        const miniPlayerToggleSpy = sinon.spy();
        const component = renderAdvancedMenu({
            miniPlayerToggleEnabled: true,
            onMiniPlayerToggle: miniPlayerToggleSpy,
        });

        component.find(Toggle).props().onToggle();
        assert.equal(miniPlayerToggleSpy.callCount, 1);
    });

    QUnit.test('on clicking Show Video Stats, invoke onStatsToggle prop', function(assert) {
        const toggleStatsSpy = sinon.spy();
        const component = renderAdvancedMenu({
            onStatsToggle: toggleStatsSpy,
        });

        component.find('.qa-show-stats-button').props().onClick();
        assert.equal(toggleStatsSpy.callCount, 1);
    });
});
