import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import CopyToClipboard from 'react-copy-to-clipboard';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { COPY_URL_FEEDBACK_DELAY, MainMenuComponent } from 'ui/components/settings/main-menu';
import { IDENTIFIER_QUALITY_MENU, IDENTIFIER_ADVANCED_MENU,
         IDENTIFIER_REPORT_ISSUES_MENU, IDENTIFIER_PLAYBACK_SPEED_MENU,
         IDENTIFIER_STAFF_MENU } from 'ui/components/settings/menu-manager';
import { PLAYER_SITE, PLAYER_CLIPS_EMBED, PLAYER_CLIPS_VIEWING } from 'util/player-type';

const DEFAULT_ARGS = Object.freeze({
    canChangePlaybackSpeed: false,
    canPopoutPlayer: true,
    captionsAvailable: false,
    currentSpeed: 1.0,
    currentTime: 0,
    isStaffUser: false,
    isVod: false,
    onMenuTransition() {},
    onPopoutPlayer() {},
    onShowCaptionsOptions() {},
    selectedQualityName: '',
    shouldShowQualityMenu: true,
    showClipsShortcutOverlay() {},
    t() {},
    vodUrlWithTimestamp: '',
    windowObj: {},
    playerType: PLAYER_SITE,
});

function renderMainMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <MainMenuComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | settings | main-menu', function() {
    QUnit.test('returns a div', function(assert) {
        const component = renderMainMenu();
        assert.equal(component.type(), 'div');
    });

    QUnit.test('does not show quality menu button if shouldShowQualityMenu is false', function(assert) {
        const component = renderMainMenu({
            shouldShowQualityMenu: false,
        });
        assert.equal(component.find('.qa-quality-button').length, 0);
    });

    QUnit.test('displays current quality in pill next to button', function(assert) {
        const fakeQuality = 'fakequality';
        const component = renderMainMenu({
            selectedQualityName: fakeQuality,
        });

        assert.equal(component.find('.qa-quality-pill').text(), fakeQuality);
    });

    QUnit.test('clicking on the quality button triggers a menu transition', function(assert) {
        const transitionSpy = sinon.spy();
        const component = renderMainMenu({
            onMenuTransition: transitionSpy,
        });

        component.find('.qa-quality-button').props().onClick();
        assert.equal(transitionSpy.callCount, 1, 'menu transition prop was invoked');
        assert.equal(
            transitionSpy.firstCall.args[0],
            IDENTIFIER_QUALITY_MENU,
            'prop was invoked with quality menu identifier'
        );
    });

    QUnit.test('clicking on the advanced button triggers a menu transition', function(assert) {
        const transitionSpy = sinon.spy();
        const component = renderMainMenu({
            onMenuTransition: transitionSpy,
        });

        component.find('.qa-advanced-button').props().onClick();
        assert.equal(transitionSpy.callCount, 1, 'menu transition prop was invoked');
        assert.equal(
            transitionSpy.firstCall.args[0],
            IDENTIFIER_ADVANCED_MENU,
            'prop was invoked with advanced menu identifier'
        );
    });

    QUnit.test('clicking on the report issue button triggers a menu transition', function(assert) {
        const transitionSpy = sinon.spy();
        const component = renderMainMenu({
            onMenuTransition: transitionSpy,
        });

        component.find('.qa-report-issue-button').props().onClick();
        assert.equal(transitionSpy.callCount, 1, 'menu transition prop was invoked');
        assert.equal(
            transitionSpy.firstCall.args[0],
            IDENTIFIER_REPORT_ISSUES_MENU,
            'prop was invoked with quality menu identifier'
        );
    });

    QUnit.module('Playback Speed Menu', function() {
        QUnit.test('show playback speed button if canChangePlaybackSpeed is true and its a vod', function(assert) {
            const component = renderMainMenu({
                canChangePlaybackSpeed: false,
                isVod: false,
            });

            assert.equal(
                component.find('.qa-playback-speed-button').length,
                0,
                'does not render playback speed button when isVod is false and canChangePlaybackSpeed is false'
            );

            component.setProps({
                canChangePlaybackSpeed: true,
                isVod: false,
            });

            assert.equal(
                component.find('.qa-playback-speed-button').length,
                0,
                'does not render playback speed button when isVod is false and canChangePlaybackSpeed is true'
            );

            component.setProps({
                canChangePlaybackSpeed: false,
                isVod: true,
            });

            assert.equal(
                component.find('.qa-playback-speed-button').length,
                0,
                'does not render playback speed button when isVod is true and canChangePlaybackSpeed is false'
            );

            component.setProps({
                canChangePlaybackSpeed: true,
                isVod: true,
            });

            assert.equal(
                component.find('.qa-playback-speed-button').length,
                1,
                'renders playback speed button when isVod is true and canChangePlaybackSpeed is true'
            );
        });

        QUnit.test('show playback speed button if canChangePlaybackSpeed is true and its a clip', function(assert) {
            const component = renderMainMenu({
                canChangePlaybackSpeed: false,
                isClip: false,
            });

            assert.equal(
                component.find('.qa-playback-speed-button').length,
                0,
                'does not render playback speed button when isClip is false and canChangePlaybackSpeed is false'
            );

            component.setProps({
                canChangePlaybackSpeed: true,
                isClip: false,
            });

            assert.equal(
                component.find('.qa-playback-speed-button').length,
                0,
                'does not render playback speed button when isClip is false and canChangePlaybackSpeed is true'
            );

            component.setProps({
                canChangePlaybackSpeed: false,
                isClip: true,
            });

            assert.equal(
                component.find('.qa-playback-speed-button').length,
                0,
                'does not render playback speed button when isClip is true and canChangePlaybackSpeed is false'
            );

            component.setProps({
                canChangePlaybackSpeed: true,
                isClip: true,
            });

            assert.equal(
                component.find('.qa-playback-speed-button').length,
                1,
                'renders playback speed button when isClip is true and canChangePlaybackSpeed is true'
            );
        });

        QUnit.test('clicking the playback speed button triggers a menu transition', function(assert) {
            const transitionSpy = sinon.spy();
            const component = renderMainMenu({
                canChangePlaybackSpeed: true,
                isVod: true,
                onMenuTransition: transitionSpy,
            });

            component.find('.qa-playback-speed-button').props().onClick();
            assert.equal(transitionSpy.callCount, 1, 'menu transition prop was invoked');
            assert.equal(
                transitionSpy.firstCall.args[0],
                IDENTIFIER_PLAYBACK_SPEED_MENU,
                'prop was invoked with playback speed menu identifier'
            );
        });

        QUnit.test('shows a pill for the current playback speed', function(assert) {
            const component = renderMainMenu({
                canChangePlaybackSpeed: true,
                currentSpeed: 2.0,
                isVod: true,
                t: str => str,
            });

            assert.equal(component.find('.qa-playback-speed-pill').text(), '2x', 'shows expected 2x string');
        });
    });

    QUnit.test('only show captions options button if captionsAvailable is true', function(assert) {
        const component = renderMainMenu({
            captionsAvailable: false,
        });

        assert.equal(component.find('.qa-captions-options-button').length, 0);

        component.setProps({
            captionsAvailable: true,
        });

        assert.equal(component.find('.qa-captions-options-button').length, 1);
    });

    QUnit.test('clicking on the show closed caption options button invokes onShowCaptionsOptions', function(assert) {
        const captionsShowSpy = sinon.spy();
        const component = renderMainMenu({
            captionsAvailable: true,
            onShowCaptionsOptions: captionsShowSpy,
        });

        component.find('.qa-captions-options-button').props().onClick();
        assert.equal(captionsShowSpy.callCount, 1);
    });

    QUnit.test('does not render popout player button if canPopoutPlayer is false', function(assert) {
        const component = renderMainMenu({
            canPopoutPlayer: false,
        });
        assert.equal(component.find('.qa-popout-player-button').length, 0);
    });

    QUnit.test('clicking on the popout player button invokes onPopoutPlayer', function(assert) {
        const popoutSpy = sinon.spy();
        const component = renderMainMenu({
            onPopoutPlayer: popoutSpy,
        });

        component.find('.qa-popout-player-button').props().onClick();
        assert.equal(popoutSpy.callCount, 1);
    });

    QUnit.test('if isVod, render a CopyToClipboard button with the vodUrlWithTimestamp', function(assert) {
        const vodUrlWithTimestamp = 'fakevideourl.com?t=0';
        const component = renderMainMenu({
            isVod: true,
            vodUrlWithTimestamp,
        });

        const clipboardButton = component.find('.qa-copy-to-clipboard-button');

        assert.equal(clipboardButton.type(), CopyToClipboard, 'is a CopyToClipboard element');
        assert.equal(clipboardButton.props().text, vodUrlWithTimestamp, 'has vodUrlWithTimestamp as the copiable text');
    });

    QUnit.test('clicking the copy url button sets state and timeout and renders `Copied!`', function(assert) {
        const timeoutId = '12345';
        const windowObj = {
            setTimeout: sinon.stub().returns(timeoutId),
            clearTimeout: sinon.spy(),
        };

        const component = renderMainMenu({
            isVod: true,
            windowObj,
        });

        const clipboardButton = component.find('.qa-copy-to-clipboard-button');

        assert.equal(component.state().hasRecentlyCopied, false, 'has not recently copied by default');
        clipboardButton.props().onSelect();
        assert.equal(component.state().hasRecentlyCopied, true, 'on copy, set hasRecentlyCopied to true');
        assert.equal(windowObj.setTimeout.callCount, 1, 'has timeout set');
        assert.equal(windowObj.setTimeout.firstCall.args[1], COPY_URL_FEEDBACK_DELAY, 'has correct delay');
        assert.equal(component.instance().copyUrlTimeoutId, timeoutId, 'updates timeoutId property');

        windowObj.setTimeout.firstCall.args[0]();

        assert.equal(component.state().hasRecentlyCopied, false, 'after delay, set hasRecentlyCopied to false');
        assert.equal(component.instance().copyUrlTimeoutId, 0, 'updates timeoutId property to 0 after timeout');
    });

    QUnit.module('Staff Menu', function() {
        QUnit.test('only renders staff menu button if isStaffUser', function(assert) {
            const component = renderMainMenu({
                isStaffUser: true,
            });

            assert.equal(component.find('.qa-staff-button').length, 1, 'staff button is shown');

            component.setProps({
                isStaffUser: false,
            });

            assert.equal(component.find('.qa-staff-button').length, 0, 'staff button is not shown');
        });

        QUnit.test('clicking the staff button triggers a menu transition', function(assert) {
            const transitionSpy = sinon.spy();
            const component = renderMainMenu({
                isStaffUser: true,
                onMenuTransition: transitionSpy,
            });

            component.find('.qa-staff-button').props().onClick();
            assert.equal(transitionSpy.callCount, 1, 'menu transition prop was invoked');
            assert.equal(
                transitionSpy.firstCall.args[0],
                IDENTIFIER_STAFF_MENU,
                'prop was invoked with staff menu identifier'
            );
        });
    });

    QUnit.module('Clips Embed Player', function() {
        QUnit.test('only renders quality, playback speed, and keyboard shortcuts', function(assert) {
            const component = renderMainMenu({
                playerType: PLAYER_CLIPS_EMBED,
                isClip: true,
                canChangePlaybackSpeed: true,
            });

            assert.equal(component.find('.qa-quality-button').length, 1, 'quality button is shown');
            assert.equal(component.find('.qa-playback-speed-button').length, 1, 'playback speed button is shown');
            assert.equal(component.find('.qa-clips-keyboard-shortcuts').length, 1, 'keyboard shortcut button is shown');

            assert.equal(component.find('.qa-captions-options-button').length, 0, 'captions is not shown');
            assert.equal(component.find('.qa-advanced-button').length, 0, 'advanced qa is not shown');
            assert.equal(component.find('.qa-report-issue-button').length, 0, 'report is not shown');
            assert.equal(component.find('.qa-popout-player-button').length, 0, 'popout is not shown');
            assert.equal(
                component.find('.qa-clip-moderation-button').length,
                0,
                'clips moderation button is not shown'
            );
            assert.equal(component.find('.qa-report-clip').length, 0, 'report clip button is not shown');
        });

        QUnit.test('show the shortcut overlay when clicked', function(assert) {
            const showClipsShortcutOverlaySpy = sinon.spy();
            const component = renderMainMenu({
                playerType: PLAYER_CLIPS_EMBED,
                isClip: true,
                showClipsShortcutOverlay: showClipsShortcutOverlaySpy,
            });

            assert.equal(showClipsShortcutOverlaySpy.callCount, 0);

            component.find('.qa-clips-keyboard-shortcuts').simulate('click');

            assert.equal(showClipsShortcutOverlaySpy.callCount, 1);
        });
    });

    QUnit.module('Clips Viewing Player', function() {
        QUnit.test('only renders quality, playback speed, keyboard shortcuts, and moderation', function(assert) {
            const component = renderMainMenu({
                playerType: PLAYER_CLIPS_VIEWING,
                isClip: true,
                canChangePlaybackSpeed: true,
                isStaffUser: true,
            });

            assert.equal(component.find('.qa-quality-button').length, 1, 'quality button is shown');
            assert.equal(component.find('.qa-playback-speed-button').length, 1, 'playback speed button is shown');
            assert.equal(component.find('.qa-clips-keyboard-shortcuts').length, 1, 'keyboard shortcut button is shown');
            assert.equal(component.find('.qa-clip-moderation-button').length, 1, 'clips moderation button is shown');
            assert.equal(component.find('.qa-report-clip').length, 1, 'report clip button is shown');

            assert.equal(component.find('.qa-captions-options-button').length, 0, 'captions is not shown');
            assert.equal(component.find('.qa-advanced-button').length, 0, 'advanced qa is not shown');
            assert.equal(component.find('.qa-report-issue-button').length, 0, 'report is not shown');
            assert.equal(component.find('.qa-popout-player-button').length, 0, 'popout is not shown');
        });

        QUnit.test('show the shortcut overlay when clicked', function(assert) {
            const showClipsShortcutOverlaySpy = sinon.spy();
            const component = renderMainMenu({
                playerType: PLAYER_CLIPS_VIEWING,
                isClip: true,
                showClipsShortcutOverlay: showClipsShortcutOverlaySpy,
            });

            assert.equal(showClipsShortcutOverlaySpy.callCount, 0);

            component.find('.qa-clips-keyboard-shortcuts').simulate('click');

            assert.equal(showClipsShortcutOverlaySpy.callCount, 1);
        });
    });
});
