import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { MenuManager, IDENTIFIER_MAIN_MENU, IDENTIFIER_QUALITY_MENU,
         IDENTIFIER_ADVANCED_MENU, IDENTIFIER_REPORT_ISSUES_MENU,
         SETTINGS_INNER_PADDING, IDENTIFIER_PLAYBACK_SPEED_MENU,
         IDENTIFIER_STAFF_MENU, IDENTIFIER_CLIP_MODERATION_MENU } from 'ui/components/settings/menu-manager';
import { MainMenu } from 'ui/containers/settings/main-menu';
import { QualityMenu } from 'ui/containers/settings/quality-menu';
import { AdvancedMenu } from 'ui/containers/settings/advanced-menu';
import { ReportIssuesMenu } from 'ui/containers/settings/report-issues-menu';
import { StaffMenu } from 'ui/containers/settings/staff-menu';
import { PlaybackSpeedMenu } from 'ui/containers/settings/playback-speed-menu';
import { ClipsModerationMenu } from 'ui/components/settings/clips-moderation-menu';

function renderMenuManager(args = {}) {
    const component = <MenuManager {...args} />;
    return shallow(component);
}

reactTest('ui | components | settings | menu-manager', function() {
    QUnit.test('renders the main menu by default', function(assert) {
        const component = renderMenuManager();
        assert.equal(component.childAt(0).type(), MainMenu, 'renders main menu');
        assert.equal(component.state('currentMenu'), IDENTIFIER_MAIN_MENU);
    });

    QUnit.test('div has the max height set to playerHeight - inner padding', function(assert) {
        const playerHeight = 500;
        const component = renderMenuManager({
            playerHeight,
        });

        assert.equal(component.props().style.maxHeight, `${playerHeight - SETTINGS_INNER_PADDING}px`);
    });

    QUnit.test('handleMenuTransition does not transition to invalid menus', function(assert) {
        const component = renderMenuManager();
        const instance = component.instance();
        const fakeIdentifier = 'fakemenuidentifier';

        instance.handleMenuTransition(fakeIdentifier);
        assert.notEqual(component.state('currentMenu'), fakeIdentifier);
    });

    QUnit.module('when currentMenu identifier is main menu', function(hooks) {
        hooks.beforeEach(function() {
            this.component = renderMenuManager();
            this.component.setState({
                currentMenu: IDENTIFIER_MAIN_MENU,
            });
        });

        QUnit.test('render MainMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), MainMenu);
        });

        QUnit.test('onMenuTransition changes menus', function(assert) {
            this.component.find(MainMenu).props().onMenuTransition(IDENTIFIER_QUALITY_MENU);
            assert.equal(this.component.state().currentMenu, IDENTIFIER_QUALITY_MENU);
        });
    });

    QUnit.module('when currentMenu identifier is clip moderation', function(hooks) {
        hooks.beforeEach(function() {
            this.component = renderMenuManager();
            this.component.setState({
                currentMenu: IDENTIFIER_CLIP_MODERATION_MENU,
            });
        });

        QUnit.test('render ClipsModerationMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), ClipsModerationMenu);
        });

        QUnit.test('onMenuTransition changes menus', function(assert) {
            this.component.find(ClipsModerationMenu).props().onMenuTransition(IDENTIFIER_QUALITY_MENU);
            assert.equal(this.component.state().currentMenu, IDENTIFIER_QUALITY_MENU);
        });
    });

    QUnit.module('when currentMenu identifier is quality menu', function(hooks) {
        hooks.beforeEach(function() {
            this.component = renderMenuManager();
            this.component.setState({
                currentMenu: IDENTIFIER_QUALITY_MENU,
            });
        });

        QUnit.test('render QualityMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), QualityMenu);
        });

        QUnit.test('onMenuTransition changes menus', function(assert) {
            this.component.find(QualityMenu).props().onMenuTransition(IDENTIFIER_MAIN_MENU);
            assert.equal(this.component.state().currentMenu, IDENTIFIER_MAIN_MENU);
        });
    });

    QUnit.module('when currentMenu identifier is advanced menu', function(hooks) {
        hooks.beforeEach(function() {
            this.component = renderMenuManager();
            this.component.setState({
                currentMenu: IDENTIFIER_ADVANCED_MENU,
            });
        });

        QUnit.test('render AdvancedMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), AdvancedMenu);
        });

        QUnit.test('onMenuTransition changes menus', function(assert) {
            this.component.find(AdvancedMenu).props().onMenuTransition(IDENTIFIER_MAIN_MENU);
            assert.equal(this.component.state().currentMenu, IDENTIFIER_MAIN_MENU);
        });
    });

    QUnit.module('when currentMenu identifier is report issues menu', function(hooks) {
        hooks.beforeEach(function() {
            this.component = renderMenuManager();
            this.component.setState({
                currentMenu: IDENTIFIER_REPORT_ISSUES_MENU,
            });
        });

        QUnit.test('render ReportIssuesMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), ReportIssuesMenu);
        });

        QUnit.test('onMenuTransition changes menus', function(assert) {
            this.component.find(ReportIssuesMenu).props().onMenuTransition(IDENTIFIER_MAIN_MENU);
            assert.equal(this.component.state().currentMenu, IDENTIFIER_MAIN_MENU);
        });
    });

    QUnit.module('when currentMenu identifier is staff menu', function(hooks) {
        hooks.beforeEach(function() {
            this.component = renderMenuManager();
            this.component.setState({
                currentMenu: IDENTIFIER_STAFF_MENU,
            });
        });

        QUnit.test('render StaffMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), StaffMenu);
        });

        QUnit.test('onMenuTransition changes menus', function(assert) {
            this.component.find(StaffMenu).props().onMenuTransition(IDENTIFIER_MAIN_MENU);
            assert.equal(this.component.state().currentMenu, IDENTIFIER_MAIN_MENU);
        });
    });

    QUnit.module('when currentMenu identifier is playback speed menu', function(hooks) {
        hooks.beforeEach(function() {
            this.component = renderMenuManager();
            this.component.setState({
                currentMenu: IDENTIFIER_PLAYBACK_SPEED_MENU,
            });
        });

        QUnit.test('render PlaybackSpeedMenu', function(assert) {
            assert.equal(this.component.childAt(0).type(), PlaybackSpeedMenu);
        });

        QUnit.test('onMenuTransition changes menus', function(assert) {
            this.component.find(PlaybackSpeedMenu).props().onMenuTransition(IDENTIFIER_MAIN_MENU);
            assert.equal(this.component.state().currentMenu, IDENTIFIER_MAIN_MENU);
        });
    });
});
