import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PLAYBACK_SPEEDS, TEXT_NORMAL, PlaybackSpeedMenuComponent } from 'ui/components/settings/playback-speed-menu';
import { IDENTIFIER_MAIN_MENU } from 'ui/components/settings/menu-manager';
import { PlaybackSpeedButton } from 'ui/components/settings/playback-speed-button';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    currentSpeed: 1.0,
    onMenuTransition() {},
    onSelectPlaybackSpeed() {},
    t() {},
});

function renderSpeedMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <PlaybackSpeedMenuComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | settings | playback-speed-menu', function() {
    QUnit.test('renders a div', function(assert) {
        const component = renderSpeedMenu();
        assert.equal(component.type(), 'div');
    });

    QUnit.test('when the header is clicked, menu transition to main menu is invoked', function(assert) {
        const transitionSpy = sinon.spy();
        const component = renderSpeedMenu({
            onMenuTransition: transitionSpy,
        });

        component.find('button.qa-header-button').simulate('click');
        assert.equal(transitionSpy.callCount, 1, 'transition prop was invoked');
        assert.equal(transitionSpy.firstCall.args[0], IDENTIFIER_MAIN_MENU, 'transition identifier is main menu');
    });

    QUnit.test('renders a PlaybackSpeedButton for every playback speed', function(assert) {
        const component = renderSpeedMenu({
            t: mockTranslateFunc,
        });
        const playbackSpeedButtons = component.find(PlaybackSpeedButton);
        assert.equal(
            playbackSpeedButtons.length,
            PLAYBACK_SPEEDS.length,
            `there are ${PLAYBACK_SPEEDS.length} buttons`
        );

        PLAYBACK_SPEEDS.map(speed => {
            const label = speed === 1.0 ? `1x (${TEXT_NORMAL})` : `${speed}x`;
            const isActive = speed === DEFAULT_ARGS.currentSpeed;
            const speedButton = (
                <PlaybackSpeedButton
                    isActive={isActive}
                    key={speed}
                    label={label}
                    onClick={DEFAULT_ARGS.onSelectPlaybackSpeed}
                    speedValue={speed}
                />
            );
            assert.ok(playbackSpeedButtons.containsMatchingElement(speedButton), `there is a button for ${label}`);
        });
    });
});
