import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { QualityMenuComponent } from 'ui/components/settings/quality-menu';
import { IDENTIFIER_MAIN_MENU } from 'ui/components/settings/menu-manager';
import { QualityButton } from 'ui/components/settings/quality-button';

const DEFAULT_ARGS = Object.freeze({
    availableQualities: [],
    selectedQuality: '',
    onMenuTransition() {},
    onQualitySelect() {},
    t: str => str,
});

function renderQualityMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <QualityMenuComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | settings | quality-menu', function() {
    QUnit.test('returns a div', function(assert) {
        const component = renderQualityMenu();
        assert.equal(component.type(), 'div');
    });

    QUnit.test('when the header is clicked, menu transition to main menu is invoked', function(assert) {
        const transitionSpy = sinon.spy();
        const component = renderQualityMenu({
            onMenuTransition: transitionSpy,
        });

        component.find('button.qa-header-button').simulate('click');
        assert.equal(transitionSpy.callCount, 1, 'transition prop was invoked');
        assert.equal(transitionSpy.firstCall.args[0], IDENTIFIER_MAIN_MENU, 'transition identifier is main menu');
    });

    QUnit.test('contains a QualityButton for every available quality', function(assert) {
        const qualities = [
            {
                group: 'low',
            },
            {
                group: 'high',
            },
        ];
        const component = renderQualityMenu({
            availableQualities: qualities,
        });

        qualities.map(quality => {
            assert.ok(component.containsMatchingElement(
                <QualityButton
                    isActive={false}
                    key={quality.group}
                    onQualitySelect={DEFAULT_ARGS.onQualitySelect}
                    qualityObj={quality}
                />
            ));
        });
    });

    QUnit.test('quality button of selectedQuality is marked as active', function(assert) {
        const selectedQuality = 'low';
        const availableQualities = [
            {
                group: selectedQuality,
            },
        ];
        const component = renderQualityMenu({
            selectedQuality,
            availableQualities,
        });

        const qualityButton = component.find(QualityButton);
        assert.ok(qualityButton.props().isActive);
    });
});
