import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ReportIssuesMenuComponent } from 'ui/components/settings/report-issues-menu';
import { IDENTIFIER_MAIN_MENU } from 'ui/components/settings/menu-manager';

const DEFAULT_ARGS = Object.freeze({
    t: str => str,
    onReportIssueSubmit() {},
    windowObj: {
        setTimeout() {},
    },
});

function renderReportIssuesMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ReportIssuesMenuComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | settings | report-issues-menu', function() {
    QUnit.test('returns a div', function(assert) {
        const component = renderReportIssuesMenu();
        assert.equal(component.type(), 'div');
    });

    QUnit.test('when a new option is selected, state is updated', function(assert) {
        const component = renderReportIssuesMenu();
        const $select = component.find('select');

        const blurEvent = {
            target: {
                value: 'fakevalue',
            },
        };

        $select.simulate('blur', blurEvent);
        assert.equal(component.state().value, blurEvent.target.value);
    });

    QUnit.test('on submit, invokes onReportIssueSubmit with issue and sets hasRecentlySubmitted', function(assert) {
        const submitHandlerSpy = sinon.spy();
        const component = renderReportIssuesMenu({
            onReportIssueSubmit: submitHandlerSpy,
        });
        const fakeIssue = 'fakeIssue';
        const fakeEvent = {
            preventDefault: sinon.spy(),
        };

        component.setState({
            value: fakeIssue,
        });

        assert.equal(component.state().hasRecentlySubmitted, false, 'has not submitted by default');
        component.find('form').simulate('submit', fakeEvent);

        assert.equal(submitHandlerSpy.callCount, 1, 'handler was invoked');
        assert.equal(submitHandlerSpy.firstCall.args[0], fakeIssue, 'handler was invoked with issue');
        assert.equal(fakeEvent.preventDefault.callCount, 1, 'event default is prevented to stop page refresh');
        assert.equal(component.state().hasRecentlySubmitted, true, 'has recently submitted');
    });

    QUnit.test('on clicking header, invokes onMenuTransition with main menu identifier', function(assert) {
        const transitionSpy = sinon.spy();
        const component = renderReportIssuesMenu({
            onMenuTransition: transitionSpy,
        });

        component.find('button.qa-header-button').simulate('click');
        assert.equal(transitionSpy.callCount, 1, 'transition handler was invoked');
        assert.equal(
            transitionSpy.firstCall.args[0],
            IDENTIFIER_MAIN_MENU,
            'handler was invoked with main menu identifier'
        );
    });

    QUnit.test('if hasRecentlySubmitted, show the text "Thanks for your report"', function(assert) {
        const component = renderReportIssuesMenu();

        assert.equal(component.find('button.qa-submit-button').text(), 'Submit', 'has submit as default text');

        component.setState({
            hasRecentlySubmitted: true,
        });

        assert.equal(
            component.find('.pl-menu__section').text(),
            'Thanks for your report',
            'shows "Thanks for your report" on submit'
        );
        assert.equal(component.find('.pl-form__input').length, 0, 'should not show a selector');
    });
});
