import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { StaffMenuComponent } from 'ui/components/settings/staff-menu';
import { IDENTIFIER_MAIN_MENU } from 'ui/components/settings/menu-manager';
import { Button } from 'ui/components/buttons/base-button';

const DEFAULT_ARGS = Object.freeze({
    onMenuTransition() {},
    onRunLocalPrerollAd() {},
    onRunLocalMidrollAd() {},
    onRunLocalPostrollAd() {},
    t() {},
});

function renderStaffMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <StaffMenuComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | settings | staff-menu', function() {
    QUnit.test('should render a div', function(assert) {
        const component = renderStaffMenu();
        assert.equal(component.type(), 'div', 'renders a div');
        assert.ok(component.hasClass('pl-menu__inner'), 'has pl-menu__inner class');
    });

    QUnit.test('when header is clicked, menu transition to main menu is invoked', function(assert) {
        const transitionSpy = sinon.spy();
        const component = renderStaffMenu({
            onMenuTransition: transitionSpy,
        });
        component.find('.qa-header-button').simulate('click');
        assert.equal(transitionSpy.callCount, 1, 'transition prop was invoked');
        assert.equal(transitionSpy.firstCall.args[0], IDENTIFIER_MAIN_MENU, 'transition identifier is main menu');
    });

    QUnit.test('contains a Run Preroll Local Ad button', function(assert) {
        const component = renderStaffMenu({});
        assert.ok(component.containsMatchingElement(
            <Button
                className="ellipsis"
                onClick={DEFAULT_ARGS.onRunLocalPrerollAd}
            >
                <span>Run Local Preroll Ad</span>
            </Button>
        ), 'has Preroll Local Ad type');
    });

    QUnit.test('ccontains a Run Midroll Local Ad button', function(assert) {
        const component = renderStaffMenu({});
        assert.ok(component.containsMatchingElement(
            <Button
                className="ellipsis"
                onClick={DEFAULT_ARGS.onRunLocalMidrollAd}
            >
                <span>Run Local Midroll Ad</span>
            </Button>
        ), 'has Midroll Local Ad type');
    });

    QUnit.test('contains a Run Postroll Local Ad button', function(assert) {
        const component = renderStaffMenu({});
        assert.ok(component.containsMatchingElement(
            <Button
                className="ellipsis"
                onClick={DEFAULT_ARGS.onRunLocalPostrollAd}
            >
                <span>Run Local Postroll Ad</span>
            </Button>
        ), 'has Postrolll Local Ad type');
    });
});
