import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { Slider } from 'ui/components/slider';
import { SliderLeft } from 'ui/components/slider/slider-left';
import { SliderThumb } from 'ui/components/slider/slider-thumb';

const DEFAULT_ARGS = Object.freeze({
    classNames: {},
    dragHandlers: {},
    max: 100,
    min: 0,
    onClick() {},
    onKeyDown() {},
    onMouseMove() {},
    onMouseOut() {},
    sliderTabIndex: 0,
    value: 0,
    windowObj: {
        requestAnimationFrame() {},
        cancelAnimationFrame() {},
    },
    valueOptimizationEnabled: false,
    getOptimizedValue() {},
});

function renderSlider(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <Slider {...args} />;
    return shallow(component);
}

reactTest('ui | components | slider', function() {
    QUnit.test('returns a div with the expected class name', function(assert) {
        const component = renderSlider({
            classNames: {
                slider: 'slider',
            },
        });

        assert.equal(component.type(), 'div');
        assert.ok(component.hasClass('slider'));
    });

    QUnit.test('passes in expected aria attributes as props', function(assert) {
        const component = renderSlider();
        assert.equal(component.props().role, 'slider');
        assert.equal(component.props()['aria-valuemin'], DEFAULT_ARGS.min);
        assert.equal(component.props()['aria-valuemax'], DEFAULT_ARGS.max);
        assert.equal(component.props()['aria-valuenow'], DEFAULT_ARGS.value);
    });

    QUnit.test('if isDragging is true, passes currentDragValue into aria-valuenow', function(assert) {
        const component = renderSlider();
        const currentDragValue = 5;
        component.setState({
            isDragging: true,
            currentDragValue,
        });

        assert.equal(component.props()['aria-valuenow'], currentDragValue);
    });

    // eslint-disable-next-line max-len
    QUnit.test('if isDragging is false and valueOptimizationEnabled is true, passes optimizedValue into aria-valuenow', function(assert) {
        const component = renderSlider({ valueOptimizationEnabled: true });
        const currentDragValue = 5;
        const optimizedValue = 10;
        component.setState({
            isDragging: false,
            currentDragValue,
            optimizedValue,
        });

        assert.equal(component.props()['aria-valuenow'], optimizedValue);
    });

    QUnit.module('SliderLeft', function() {
        QUnit.test('contains a SliderLeft component with passed in args', function(assert) {
            const args = {
                min: 5,
                max: 10,
                value: 7,
                classNames: {
                    slider: 'slider',
                    sliderLeft: 'slider-left',
                },
            };

            const component = renderSlider(args);
            assert.ok(component.containsMatchingElement(
                <SliderLeft
                    min={args.min}
                    max={args.max}
                    value={args.value}
                    className="slider-left"
                />
            ));
        });

        QUnit.test('if isDragging is true, SliderLeft uses the currentDragValue for value', function(assert) {
            const component = renderSlider();
            const currentDragValue = 5;
            component.setState({
                isDragging: true,
                currentDragValue,
            });

            assert.ok(component.containsMatchingElement(
                <SliderLeft
                    min={DEFAULT_ARGS.min}
                    max={DEFAULT_ARGS.max}
                    value={currentDragValue}
                    className=""
                />
            ));
        });

        // eslint-disable-next-line max-len
        QUnit.test('if isDragging is false, and valueOptimizationEnabled is true, SliderLeft uses the optimizedValue for value', function(assert) {
            const component = renderSlider({ valueOptimizationEnabled: true });
            const currentDragValue = 5;
            const optimizedValue = 10;
            component.setState({
                isDragging: false,
                currentDragValue,
                optimizedValue,
            });

            assert.ok(component.containsMatchingElement(
                <SliderLeft
                    min={DEFAULT_ARGS.min}
                    max={DEFAULT_ARGS.max}
                    value={optimizedValue}
                    className=""
                />
            ));
        });
    });

    QUnit.module('SliderThumb', function() {
        QUnit.test('contains a SliderThumb component with passed in args', function(assert) {
            const args = {
                min: 5,
                max: 10,
                value: 7,
                classNames: {
                    slider: 'slider',
                    sliderLeft: 'slider-left',
                    sliderThumb: 'slider-thumb',
                },
            };

            const component = renderSlider(args);
            assert.ok(component.containsMatchingElement(
                <SliderThumb
                    className="slider-thumb"
                    value={args.value}
                    min={args.min}
                    max={args.max}
                />
            ));
        });

        QUnit.test('if isDragging is true, SliderThumb uses the currentDragValue for value', function(assert) {
            const component = renderSlider();
            const currentDragValue = 5;
            component.setState({
                isDragging: true,
                currentDragValue,
            });

            assert.ok(component.containsMatchingElement(
                <SliderThumb
                    className=""
                    value={currentDragValue}
                    min={DEFAULT_ARGS.min}
                    max={DEFAULT_ARGS.max}
                />
            ));
        });

        // eslint-disable-next-line max-len
        QUnit.test('if isDragging is false, and valueOptimizationEnabled is true, SliderThumb uses the optimizedValue for value', function(assert) {
            const component = renderSlider({ valueOptimizationEnabled: true });
            const currentDragValue = 5;
            const optimizedValue = 10;
            component.setState({
                isDragging: false,
                currentDragValue,
                optimizedValue,
            });

            assert.ok(component.containsMatchingElement(
                <SliderThumb
                    className=""
                    value={optimizedValue}
                    min={DEFAULT_ARGS.min}
                    max={DEFAULT_ARGS.max}
                />
            ));
        });
    });

    QUnit.test('attaches onKeyDown prop to slider', function(assert) {
        const onKeyDownHandler = sinon.spy();

        const component = renderSlider({
            onKeyDown: onKeyDownHandler,
        });

        component.simulate('keydown');
        assert.ok(onKeyDownHandler.called);
    });

    QUnit.test('attaches onMouseMove prop to slider', function(assert) {
        const handler = sinon.spy();

        const component = renderSlider({
            onMouseMove: handler,
        });

        component.simulate('mousemove');
        assert.ok(handler.called);
    });

    QUnit.test('attaches onMouseOut prop to slider', function(assert) {
        const handler = sinon.spy();

        const component = renderSlider({
            onMouseOut: handler,
        });

        component.simulate('mouseout');
        assert.ok(handler.called);
    });

    QUnit.test('calculate position and call onclick handler on click', function(assert) {
        const onClickHandler = sinon.spy();
        const clickEventData = {
            pageX: 15,
        };
        const fakeValue = 5;

        const component = renderSlider({
            onClick: onClickHandler,
        });

        const instance = component.instance();
        sinon.stub(instance, 'calculateValueAtPosition').returns(fakeValue);

        component.simulate('click', clickEventData);
        assert.ok(instance.calculateValueAtPosition.calledWith(clickEventData.pageX));
        assert.ok(onClickHandler.calledWith(fakeValue));
    });

    QUnit.test('calculateValueAtPosition calculates value for given point', function(assert) {
        const fakeClientRect = {
            left: 0,
            width: 100,
        };

        const fakeSliderRef = {
            getBoundingClientRect() {
                return fakeClientRect;
            },
        };

        const pageX = 5;

        const args = {
            min: 0,
            max: 100,
            value: 50,
        };

        const component = renderSlider(args);
        const instance = component.instance();

        instance.$slider = fakeSliderRef;

        const actual = instance.calculateValueAtPosition(pageX);
        const expected = (pageX - fakeClientRect.left) / fakeClientRect.width * (args.max - args.min) + args.min;
        assert.equal(actual, expected);
    });

    QUnit.test('renders behindSliderChildren prop', function(assert) {
        const child = <div className="child" />;
        const component = renderSlider({
            behindSliderChildren: child,
        });

        assert.ok(component.childAt(0).hasClass('child'));
        assert.equal(component.childAt(1).type(), SliderLeft);
    });

    QUnit.test('renders afterSliderChildren prop', function(assert) {
        const child = <div className="child" />;
        const component = renderSlider({
            afterSliderChildren: child,
        });

        assert.equal(component.childAt(0).type(), SliderLeft);
        assert.ok(component.childAt(1).hasClass('child'));
    });

    QUnit.test('passes in sliderTabIndex prop for tabindex value', function(assert) {
        const args = {
            sliderTabIndex: 3,
        };

        const component = renderSlider(args);
        assert.equal(component.prop('tabIndex'), args.sliderTabIndex);
    });

    QUnit.test('on mousedown, sets isDragging, currentDragValue and adds mousemove handlers', function(assert) {
        const eventData = {
            pageX: 5,
        };
        const addListenerSpy = sinon.spy();
        const fakeSliderRef = {
            getBoundingClientRect() {
                return {
                    left: 0,
                    width: 100,
                };
            },
            ownerDocument: {
                addEventListener: addListenerSpy,
            },
        };

        const component = renderSlider();
        const instance = component.instance();

        instance.$slider = fakeSliderRef;

        assert.notOk(component.state().isDragging, 'isDragging is defaulted to false');

        component.simulate('mousedown', eventData);

        assert.ok(component.state().isDragging, 'on mousedown, isDragging is set to true');
        assert.equal(
            component.state().currentDragValue,
            instance.calculateValueAtPosition(eventData.pageX),
            'sets currentDragValue'
        );
        assert.ok(addListenerSpy.calledWith('mouseup', instance), 'mouseup listener set on document');
        assert.ok(addListenerSpy.calledWith('mousemove', instance), 'mousemove listener set on document');
    });

    QUnit.test('on touchstart, set isDragging, currentDragValue, identifier and adds touch handlers', function(assert) {
        const touchIdentifier = QUnit.config.current.testId;
        const eventData = {
            touches: [
                {
                    pageX: 5,
                    identifier: touchIdentifier,
                },
            ],
        };
        const addListenerSpy = sinon.spy();
        const fakeSliderRef = {
            getBoundingClientRect() {
                return {
                    left: 0,
                    width: 100,
                };
            },
            ownerDocument: {
                addEventListener: addListenerSpy,
            },
        };

        const component = renderSlider();
        const instance = component.instance();

        instance.$slider = fakeSliderRef;

        assert.notOk(component.state().isDragging, 'isDragging is defaulted to false');

        component.simulate('touchstart', eventData);

        assert.ok(component.state().isDragging, 'on touchstart, isDragging is set to true');
        assert.equal(component.state().touchIdentifier, touchIdentifier, 'touch identifier is set');
        assert.equal(
            component.state().currentDragValue,
            instance.calculateValueAtPosition(eventData.touches[0].pageX),
            'sets currentDragValue'
        );
        assert.ok(addListenerSpy.calledWith('touchmove', instance), 'mouseup listener set on document');
        assert.ok(addListenerSpy.calledWith('touchend', instance), 'mousemove listener set on document');
        assert.ok(addListenerSpy.calledWith('touchcancel', instance), 'mousemove listener set on document');
    });

    QUnit.test('handleTouchEnd removes document event listeners', function(assert) {
        const touchIdentifier = QUnit.config.current.testId;
        const fakeTouchEvent = {
            changedTouches: [{ identifier: touchIdentifier }],
        };
        const component = renderSlider();
        const instance = component.instance();
        const removeListenerSpy = sinon.spy();
        const fakeSliderRef = {
            getBoundingClientRect() {
                return {
                    left: 0,
                    width: 100,
                };
            },
            ownerDocument: {
                removeEventListener: removeListenerSpy,
            },
        };

        instance.$slider = fakeSliderRef;

        component.setState({
            touchIdentifier,
        });

        instance.handleTouchEnd(fakeTouchEvent);
        assert.ok(removeListenerSpy.calledWith('touchmove', instance), 'removes touchmove listener');
        assert.ok(removeListenerSpy.calledWith('touchend', instance), 'removes touchend listener');
        assert.ok(removeListenerSpy.calledWith('touchcancel', instance), 'removes touchcancel listener');
    });

    QUnit.test('handleMouseUp removes document event listeners', function(assert) {
        const fakeMouseEvent = {
            pageX: 5,
        };
        const component = renderSlider();
        const instance = component.instance();
        const removeListenerSpy = sinon.spy();
        const fakeSliderRef = {
            getBoundingClientRect() {
                return {
                    left: 0,
                    width: 100,
                };
            },
            ownerDocument: {
                removeEventListener: removeListenerSpy,
            },
        };

        instance.$slider = fakeSliderRef;

        instance.handleMouseUp(fakeMouseEvent);
        assert.ok(removeListenerSpy.calledWith('mousemove', instance), 'removes mousemove listener');
        assert.ok(removeListenerSpy.calledWith('mouseup', instance), 'removes mouseup listener');
    });

    QUnit.test('touchstart calls onDragStart', function(assert) {
        const touchEventData = {
            touches: [{ pageX: 5 }],
        };

        const fakeSliderRef = {
            getBoundingClientRect() {
                return {};
            },
            ownerDocument: {
                addEventListener() {},
            },
        };

        const component = renderSlider();
        const instance = component.instance();
        instance.$slider = fakeSliderRef;

        sinon.spy(instance, 'onDragStart');
        component.simulate('touchstart', touchEventData);

        assert.ok(instance.onDragStart.called);
    });

    QUnit.test('mousedown calls onDragStart', function(assert) {
        const mouseEventData = {
            pageX: 5,
        };
        const fakeSliderRef = {
            getBoundingClientRect() {
                return {};
            },
            ownerDocument: {
                addEventListener() {},
            },
        };

        const component = renderSlider();
        const instance = component.instance();
        instance.$slider = fakeSliderRef;

        sinon.spy(instance, 'onDragStart');
        component.simulate('mousedown', mouseEventData);
        assert.ok(instance.onDragStart.called);
    });

    QUnit.test('handleEvent delegates events appropriately', function(assert) {
        const touchIdentifier = QUnit.config.current.testId;
        const component = renderSlider();

        component.setState({
            touchIdentifier,
        });

        const instance = component.instance();

        const fakeSliderRef = {
            getBoundingClientRect() {
                return {};
            },
            ownerDocument: {
                addEventListener() {},
                removeEventListener() {},
            },
        };
        instance.$slider = fakeSliderRef;

        sinon.spy(instance, 'onDragEvent');
        sinon.spy(instance, 'onDragEnd');

        // touchmove
        sinon.spy(instance, 'handleTouchMove');
        instance.handleEvent({
            type: 'touchmove',
            touches: [
                { identifier: touchIdentifier },
            ],
        });
        assert.ok(instance.handleTouchMove.called, 'touchmove calls handleTouchMove');
        assert.ok(instance.onDragEvent.called, 'touchmove calls onDragEvent');

        // mousemove
        instance.onDragEvent.reset();
        sinon.spy(instance, 'handleMouseMove');
        instance.handleEvent({
            type: 'mousemove',
            pageX: 5,
        });
        assert.ok(instance.handleMouseMove.called, 'mousemove calls handleMouseMove');
        assert.ok(instance.onDragEvent.called, 'mousemove calls onDragEvent');

        // touchend
        sinon.spy(instance, 'handleTouchEnd');
        instance.handleEvent({
            type: 'touchend',
            changedTouches: [
                { identifier: touchIdentifier },
            ],
        });
        assert.ok(instance.handleTouchEnd.called, 'touchend calls handleTouchEnd');
        assert.ok(instance.onDragEnd.called, 'touchend calls onDragEnd');

        // touchcancel
        instance.handleTouchEnd.reset();
        instance.onDragEnd.reset();
        component.setState({ touchIdentifier });
        instance.handleEvent({
            type: 'touchcancel',
            changedTouches: [
                { identifier: touchIdentifier },
            ],
        });
        assert.ok(instance.handleTouchEnd.called, 'touchcancel calls handleTouchEnd');
        assert.ok(instance.onDragEnd.called, 'touchcancel calls onDragEnd');

        // mouseup
        sinon.spy(instance, 'handleMouseUp');
        instance.onDragEnd.reset();
        instance.handleEvent({
            type: 'mouseup',
            pageX: 5,
        });
        assert.ok(instance.handleMouseUp.called, 'mouseup calls handleMouseUp');
        assert.ok(instance.onDragEnd.called, 'mouseup calls onDragEnd');
    });

    // eslint-disable-next-line max-len
    QUnit.test('if valueOptimizationEnabled is true, sets an RAF on mount to update optimized value', function(assert) {
        const optimizedValue1 = 10;
        const optimizedValue2 = 20;
        const requestAnimationID1 = 12314;
        const requestAnimationID2 = 23141251;
        const windowObj = {
            requestAnimationFrame: sinon.stub().returns(requestAnimationID1),
        };

        const component = renderSlider({
            getOptimizedValue: () => optimizedValue1,
            valueOptimizationEnabled: true,
            windowObj,
        });

        component.instance().componentDidMount();

        assert.equal(
            component.state().optimizedValue,
            optimizedValue1,
            'on mount retrieves and updated optimized value'
        );
        assert.equal(
            component.state().requestAnimationID,
            requestAnimationID1,
            'on mount sets an RAF handler'
        );

        const rafHandler = windowObj.requestAnimationFrame.firstCall.args[0];

        component.setProps({
            getOptimizedValue: () => optimizedValue2,
        });

        windowObj.requestAnimationFrame = () => requestAnimationID2;

        rafHandler();

        assert.equal(
            component.state().optimizedValue,
            optimizedValue2,
            'the raf handler retrieves and updates the optimized value'
        );
        assert.equal(
            component.state().requestAnimationID,
            requestAnimationID2,
            'the raf handler sets another raf handler'
        );
    });

    QUnit.test('on componentWillUnmount calls cancelAnimationFrame with requestAnimationID', function(assert) {
        const requestAnimationID = 123456;
        const windowObj = {
            requestAnimationFrame() {},
            cancelAnimationFrame: sinon.spy(),
        };

        const component = renderSlider({
            valueOptimizationEnabled: true,
            windowObj,
        });

        component.setState({
            requestAnimationID,
        });

        assert.equal(windowObj.cancelAnimationFrame.callCount, 0);

        component.instance().componentWillUnmount();

        assert.equal(windowObj.cancelAnimationFrame.callCount, 1);
        assert.equal(windowObj.cancelAnimationFrame.firstCall.args[0], requestAnimationID);
    });
});
