import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { Stats } from 'ui/components/stats/stats';

const DEFAULT_STATS = Object.freeze({
    bufferSize: 0,
    displayResolution: '',
    skippedFrames: 0,
    fps: 0,
    hlsLatencyBroadcaster: 0,
    hlsLatencyEncoder: 0,
    memoryUsage: 0,
    playbackRate: 0,
    videoResolution: '',
    backendVersion: '',
});

const mapStats = statsProps => {
    return statsProps.reduce((map, props) => {
        // eslint-disable-next-line no-param-reassign
        map[props.statName] = props.value;
        return map;
    }, {});
};

const DEFAULT_ARGS = Object.freeze({
    stats: DEFAULT_STATS,
    isLive: true,
    onClose() {},
});

const mockStats = Object.freeze({
    bufferSize: 1234,
    displayResolution: '1243x213',
    skippedFrames: 123124,
    fps: 123424,
    hlsLatencyBroadcaster: 1231414,
    hlsLatencyEncoder: 43535,
    memoryUsage: 5233,
    playbackRate: 243653,
    videoResolution: '432x2342',
    backendVersion: '11.1.1.1',
});

function renderStats(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <Stats {...args} />;
    return shallow(component);
}

reactTest('ui | components | stats | stats', function() {
    QUnit.test('renders a ul', function(assert) {
        const component = renderStats();
        assert.equal(component.type(), 'ul');
    });

    QUnit.test('onClose is invoked when close button is clicked', function(assert) {
        const onClose = sinon.spy();
        const component = renderStats({
            onClose,
        });

        component.find('.player-button--close').simulate('click');

        assert.equal(onClose.callCount, 1);
    });

    QUnit.test('renders correct values from stats object - live', function(assert) {
        const component = renderStats({
            stats: mockStats,
        });

        const expectedValues = {
            'Video Resolution': mockStats.videoResolution,
            'Display Resolution': mockStats.displayResolution,
            FPS: mockStats.fps,
            'Skipped Frames': mockStats.skippedFrames,
            'Buffer Size': `${mockStats.bufferSize} sec.`,
            'Latency To Broadcaster': `${mockStats.hlsLatencyBroadcaster} sec.`,
            'Latency To Encoder': `${mockStats.hlsLatencyEncoder} sec.`,
            'Playback Rate': `${mockStats.playbackRate} Kbps`,
            'Memory Usage': mockStats.memoryUsage,
            'Backend Version': mockStats.backendVersion,
        };

        const renderedStats = component.children().
            map(child => child.props()).
            filter(child => child.statName);

        assert.deepEqual(mapStats(renderedStats), expectedValues);
    });

    QUnit.test('renders correct values from stats object - vod', function(assert) {
        const component = renderStats({
            stats: mockStats,
            isLive: false,
        });

        const expectedValues = {
            'Video Resolution': mockStats.videoResolution,
            'Display Resolution': mockStats.displayResolution,
            FPS: mockStats.fps,
            'Skipped Frames': mockStats.skippedFrames,
            'Buffer Size': `${mockStats.bufferSize} sec.`,
            'Playback Rate': `${mockStats.playbackRate} Kbps`,
            'Memory Usage': mockStats.memoryUsage,
            'Backend Version': mockStats.backendVersion,
        };

        const renderedStats = component.children().
            map(child => child.props()).
            filter(child => child.statName);

        assert.deepEqual(mapStats(renderedStats), expectedValues);
    });
});
