import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { StreamInfoDisplayComponent } from 'ui/components/stream-info/stream-info-display';
import { I18N_INSTANCE } from 'lang/i18n';
import { Interpolate } from 'react-i18next';

const DEFAULT_ARGS = Object.freeze({
    gameName: '',
    gameUrl: '',
    lang: {},
    live: false,
    name: '',
    streamUrl: '',
    title: '',
    viewerCount: 0,
    isMobile: false,
    i18n: I18N_INSTANCE, // needed for <Interpolate> component
});

function renderStreamInfoDisplay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <StreamInfoDisplayComponent {...args} />;
    return shallow(component);
}

reactTest('ui | components | stream-info | stream-info-display', function() {
    QUnit.module('given default input', function(hooks) {
        hooks.beforeEach(function() {
            this.streamInfo = renderStreamInfoDisplay();
        });

        QUnit.test('returns expected top level structure', function(assert) {
            assert.equal(this.streamInfo.type(), 'div', 'should generate a div element');
            assert.equal(this.streamInfo.children().length, 3, 'top level div should have 3 children');
        });

        QUnit.test('returns expected username child', function(assert) {
            const userName = this.streamInfo.childAt(0);
            assert.equal(userName.type(), 'div', 'username should be a div');
            assert.ok(userName.hasClass('player-streaminfo__name'), 'username <div> should have expected class');
            assert.equal(userName.children().length, 1, 'username div should have 1 child');
            const nameAnchor = userName.childAt(0);
            assert.ok(
                nameAnchor.hasClass('player-text-link player-text-link--no-color'),
                'name anchor should have expected class'
            );
            assert.equal(nameAnchor.prop('href'), DEFAULT_ARGS.streamUrl, 'name anchor should have expected href');
            assert.equal(nameAnchor.prop('target'), '_blank', 'name anchor should have expected target');
            assert.equal(nameAnchor.prop('onClick'), undefined, 'name anchor should not have onClick defined');
            assert.equal(nameAnchor.text(), DEFAULT_ARGS.name, 'name anchor should have expected text');
        });

        QUnit.test('returns expected title child', function(assert) {
            const title = this.streamInfo.childAt(1);
            assert.equal(title.type(), 'div', 'title should be a div');
            assert.ok(
                title.hasClass('player-streaminfo__title qa-stream-title'),
                'title <div> should have expected class'
            );
            assert.equal(title.children().length, 0, 'title div should have 0 children');
            assert.equal(title.text(), '', 'title should have expected text');
        });

        QUnit.test('returns expected live data child', function(assert) {
            const liveData = this.streamInfo.childAt(2);
            assert.equal(liveData.type(), 'div', 'live data should be a div');
            assert.ok(liveData.hasClass('player-streaminfo__game'), 'liveData <div> should have expected class');
            assert.equal(liveData.children().length, 0, 'liveData div should have no children (since live is false)');
        });
    });

    QUnit.module('given input', function(hooks) {
        hooks.beforeEach(function() {
            this.input = {
                className: 'some-class',
                gameName: 'gameName',
                gameUrl: 'https://game.url',
                lang: I18N_INSTANCE,
                live: true,
                name: 'name',
                onClick: () => {},
                streamUrl: 'https://stream.url',
                title: 'title',
                viewerCount: 12345,
                i18n: I18N_INSTANCE,
                isMobile: false,
            };
            this.streamInfo = renderStreamInfoDisplay(this.input);
        });

        QUnit.test('hides streaminfo__title if live and mobile', function(assert) {
            const component = renderStreamInfoDisplay({
                live: true,
                isMobile: true,
            });

            const title = component.find('.player-streaminfo__title');
            assert.ok(title.hasClass('player-streaminfo__title--hide'), 'title is hidden');
        });

        QUnit.test('does not streaminfo__title if live and not mobile', function(assert) {
            const component = renderStreamInfoDisplay({
                live: true,
                isMobile: false,
            });

            const title = component.find('.player-streaminfo__title');
            assert.notOk(title.hasClass('hide'), 'hide class is not on title component');
        });

        QUnit.test('does not streaminfo--title if not live and mobile', function(assert) {
            const component = renderStreamInfoDisplay({
                live: false,
                isMobile: true,
            });

            const title = component.find('.player-streaminfo__title');
            assert.notOk(title.hasClass('hide'), 'hide class is not on title component');
        });

        QUnit.test('returns expected top level structure', function(assert) {
            assert.equal(this.streamInfo.type(), 'div', 'should generate a div element');
            assert.ok(this.streamInfo.hasClass('some-class'), 'applies the passed className');
            assert.equal(this.streamInfo.children().length, 3, 'top level div should have 3 children');
        });

        QUnit.test('returns expected username child', function(assert) {
            const userName = this.streamInfo.childAt(0);
            assert.equal(userName.type(), 'div', 'username should be a div');
            assert.ok(userName.hasClass('player-streaminfo__name'), 'username <div> should have expected class');
            assert.equal(userName.children().length, 1, 'username div should have 1 child');
            const nameAnchor = userName.childAt(0);
            assert.ok(
                nameAnchor.hasClass('player-text-link player-text-link--no-color qa-display-name'),
                'name anchor should have expected class'
            );
            assert.equal(nameAnchor.prop('href'), this.input.streamUrl, 'name anchor should have expected href');
            assert.equal(nameAnchor.prop('target'), '_blank', 'name anchor should have expected target');
            assert.equal(nameAnchor.prop('onClick'), this.input.onClick, 'name anchor should given onClick function');
            assert.equal(nameAnchor.text(), this.input.name, 'name anchor should have expected text');
        });

        QUnit.test('returns expected title child', function(assert) {
            const title = this.streamInfo.childAt(1);
            assert.equal(title.type(), 'div', 'title should be a div');
            assert.ok(
                title.hasClass('player-streaminfo__title'),
                'title <div> should have expected class'
            );
            // note: text is considered children
            assert.equal(title.children().length, 1, 'title div should have 1 child');
            assert.equal(title.text(), this.input.title, 'title should have expected text');
        });

        QUnit.test('returns expected live data child with viewer info', function(assert) {
            const liveData = this.streamInfo.childAt(2);
            assert.equal(liveData.type(), 'div', 'live data should be a div');
            assert.ok(
                liveData.hasClass('player-streaminfo__game'),
                'live data <div> should have expected class'
            );
            assert.equal(liveData.children().length, 1, 'live data div should have 1 child');
            const viewers = liveData.childAt(0);
            assert.equal(viewers.type(), 'div', 'viewers should be a div');
            assert.ok(
                viewers.hasClass('player-streaminfo__viewers'),
                'viewers <div> should have expected class'
            );
            const gameInfoComponent = (
                <a
                    className="player-text-link player-text-link--no-color qa-game-name"
                    href={this.input.gameUrl}
                    target="_blank"
                >
                    {this.input.gameName}
                </a>
            );
            const localeViewerCount = Number(this.input.viewerCount).toLocaleString(this.input.lang.shortCode);
            const viewerCountComponent = (
                <span>
                    {localeViewerCount}
                </span>
            );
            const expectedComponent = (
                <Interpolate
                    i18nKey={'playing {{game}} for {{viewerCount}} viewer'}
                    game={gameInfoComponent}
                    viewerCount={viewerCountComponent}
                    options={{
                        count: this.input.viewerCount,
                    }}
                />
            );

            assert.ok(
                viewers.containsMatchingElement(expectedComponent),
                'should contain expected Interpolated component'
            );
        });
    });
});
